using UnityEngine;
using System.Collections;
using System.Collections.Generic;

public class InputManager : MonoBehaviour
{
	// Instance
	private static InputManager m_Instance;
	public static InputManager Instance
	{
		get
		{
			if(m_Instance == null)
			{
				m_Instance = FindObjectOfType(typeof(InputManager)) as InputManager;
			}
			
			return m_Instance;
		}
	}
	// Devices used
	private List<DeviceController>[] m_DevicesByPlayer;
	private List<DeviceController> m_AllDevices;
	
	#region Initialisation
	void Awake()
	{
		// Creation of all devices list
		m_AllDevices = new List<DeviceController>();
		// Creation for players devices lists
		m_DevicesByPlayer = new List<DeviceController>[InputSettings.MaxPlayers];
		for(int i=0; i<InputSettings.MaxPlayers; ++i)
		{
			m_DevicesByPlayer[i] = new List<DeviceController>();
		}
		
#if UNITY_ANDROID
		DeviceController fingers = DeviceControllerFingers.Connection();
		fingers.IsUsed = true;
		m_DevicesByPlayer[0].Add(fingers);
		m_AllDevices.Add(fingers);
#else
		// Creates a controller for the keyboard
		DeviceController keyboard = DeviceControllerKeyboard.Connection();
		keyboard.IsUsed = true;
		m_DevicesByPlayer[0].Add(keyboard);
		m_AllDevices.Add(keyboard);
#endif
	}
	#endregion
	
	#region Update
	void Update()
	{
		foreach(DeviceController device in m_AllDevices)
		{
			device.UpdateController();
			device.UpdateAxis();
			device.UpdateActions();
			device.UpdateMenuInputs();
		}
		
		CheckDevicesConnected();
	}
	
	private void CheckDevicesConnected()
	{
		// TODO : Connection des nouveaux périphériques
	}
	#endregion
	
	#region Utils
	private bool PlayerHasDevices(int player)
	{
		if(player < 1 
		|| player > InputSettings.MaxPlayers
		|| player > m_DevicesByPlayer.Length)
		{
			return false;
		}
		
		return true;
	}
	#endregion
	
	#region Controllers
	// Getter
	
	
	// Setter
	
	#endregion
	
	#region Movements
	// Getter
	public float GetAxisValue(InputSettings.Movements axis)
	{
		return GetAxisValue(1,axis);
	}
	
	public float GetAxisValue(int player, InputSettings.Movements axis)
	{
		if(!PlayerHasDevices(player))
			return 0f;
		
		foreach(DeviceController device in m_DevicesByPlayer[player-1])
		{
			if(device.IsUsed)
			{
				return device.GetAxisValue(axis);
			}
		}
		
		return 0f;
	}
	
	// Setter
	
	#endregion
	
	#region Actions
	// Getter
	public bool GetActionStatus(InputSettings.Actions action)
	{
		return GetActionStatus(1,action,InputSettings.ActionStatus.Down);
	}
	
	public bool GetActionStatus(InputSettings.Actions action, InputSettings.ActionStatus status)
	{
		return GetActionStatus(1,action,status);
	}
	
	public bool GetActionStatus(int player, InputSettings.Actions action, InputSettings.ActionStatus status)
	{
		if(!PlayerHasDevices(player))
			return false;
		
		foreach(DeviceController device in m_DevicesByPlayer[player-1])
		{
			if(device.IsUsed)
			{
				return device.GetActionStatus(action,status);
			}
		}
		
		return false;
	}
	
	// Setter
	
	#endregion
	
	#region Menu inputs
	// Getter
	public bool GetMenuInputStatus(InputSettings.MenuInputs menuInput)
	{
		return GetMenuInputStatus(1,menuInput);
	}
	
	public bool GetMenuInputStatus(int player, InputSettings.MenuInputs menuInput)
	{
		if(!PlayerHasDevices(player))
			return false;
		
		foreach(DeviceController device in m_DevicesByPlayer[player-1])
		{
			if(device.IsUsed)
			{
				bool res = device.GetMenuInputStatus(menuInput);
				return res;
			}
		}
		
		return false;
	}
	
	public Vector3 GetMousePosition()
	{
		return GetMousePosition(1);
	}
	
	public Vector3 GetMousePosition(int player)
	{
		if(!PlayerHasDevices(player))
			return Vector3.zero;
		
		foreach(DeviceController device in m_DevicesByPlayer[player-1])
		{
			if(device.IsUsed)
			{
				Vector3 res = device.GetMousePosition();
				return res;
			}
		}
		
		return Vector3.zero;
	}

    public bool GetMouseClic(int clic)
    {
        return GetMouseClic(1, clic);
    }

    public bool GetMouseClic(int player, int clic)
    {
        if (!PlayerHasDevices(player))
            return false;

        foreach (DeviceController device in m_DevicesByPlayer[player - 1])
        {
            if (device.IsUsed)
            {
                return device.GetMouseClic(clic);
            }
        }

        return false;
    }
	
	// Setter
	
	#endregion
}