﻿using UnityEngine;
using System.Collections;
using System.Collections.Generic;
using System;

public class GameManager : MonoBehaviour
{
    private static GameManager s_Instance;
    public static GameManager Instance
    {
        get
        {
            if (s_Instance == null)
            {
                GameObject go = Instantiate(Resources.Load("Prefabs/_GameManager")) as GameObject;
                go.name = "_GameManager";
            }
            return s_Instance;
        }
    }

    // Game management
    private bool m_InGame;

    // Menu
    private Menu m_Menu;

    // Characters
    // -> Hero
    private Character m_Hero;
    private Transform m_HeroTransform;
    public Transform HeroTransform
    {
        get { return m_HeroTransform; }
    }
    public bool PlayerCanMove
    {
        set 
        {
            m_Hero.CharacterCanMove(value);
        }
    }
    // -> Enemies
    private List<Character> m_Enemies;

    // Progress
    private Level m_CurrentLevel;
    private ProgressData m_ProgressData;
    // HACK !!!!
    public ProgressData GameData
    {
        get { return m_ProgressData; }
    }

    // Save
    private SaveManager m_SaveManager;

    // Sounds
    private AudioSource m_AudioSource;
    public bool CanPlayMusic;
    public AudioClip m_Music;

    #region Initialization
    void Awake()
    {
#if UNITY_WEBPLAYER
        QualitySettings.SetQualityLevel(4); // beautiful
#endif

        DontDestroyOnLoad(gameObject);
        s_Instance = this;
        m_InGame = false;

        m_ProgressData = new ProgressData();
        m_SaveManager = new SaveManager(m_ProgressData);

        m_AudioSource = GetComponent<AudioSource>();

        StartCoroutine("C_Awake");
	}

    private IEnumerator C_Awake()
    {
        yield return 0;
        m_SaveManager.Load();

        if (Application.loadedLevelName == "MainMenu")
        {
            StartCoroutine("C_InitializeMainMenu");
        }
        else if (Application.loadedLevelName == "End")
        {
            StartCoroutine("C_InitializeEnd");
        }
        else
        {
            // Check for the current level
            string[] lvlNames = Enum.GetNames(typeof(LevelSettings.LevelName));
            for (int i = 0; i < lvlNames.Length; ++i)
            {
                if (lvlNames[i] == Application.loadedLevelName)
                {
                    if (m_ProgressData.CurrentLevelName != (LevelSettings.LevelName)i)
                    {
                        m_ProgressData.HackLaunch();
                    }
                    m_ProgressData.SetCurrentLevelName((LevelSettings.LevelName)i);
                }
            }

            StartCoroutine("C_InitializeLevel");
        }
    }


    // MAIN MENU
    private IEnumerator C_InitializeMainMenu()
    {
        // Wait for loading
        while (Application.isLoadingLevel)
        {
            yield return 0;
        }

        // Get the main menu instance
        m_Menu = FindObjectOfType(typeof(Menu)) as Menu;
        m_Menu.Initialize();

        if (Fade.Instance.Opaque)
        {
            Fade.Instance.Out(0.5f);
            yield return new WaitForSeconds(0.5f);
        }
    }

    // LEVEL
    private IEnumerator C_InitializeLevel()
    {
        // Wait for loading
        while (Application.isLoadingLevel)
        {
            yield return 0;
        }

        // Get and initialize the level
        m_CurrentLevel = (FindObjectOfType(typeof(Level)) as Level);
        if (m_ProgressData.LastCheckPoint == null) // Level begin
        {
            m_CurrentLevel.SetStoryProgress(0);
            m_ProgressData.NewLevel();
        }
        else // a save is available
        {
            // Set the progress for all level checkpoint
            m_ProgressData.SetProgressForAllCheckpoints(m_CurrentLevel.CheckPoints);
            // Set the story progress (with the last check point)
            m_CurrentLevel.SetStoryProgress(m_ProgressData.LastCheckPoint.StoryStep);
        }

        m_SaveManager.Save();
        m_ProgressData.LevelBegin();

        GameObject go = GameObject.Find("Hero");
        if (go != null)
        {
            m_Hero = go.GetComponent<Hero>();
            m_HeroTransform = m_Hero.m_CharacterModel;
            if(m_ProgressData.CurrentCheckPoint != null)
                m_Hero.transform.position = m_ProgressData.CurrentCheckPoint.Position;
        }

        Character[] tmpArray = FindObjectsOfType(typeof(Enemy)) as Character[];
        m_Enemies = new List<Character>();
        if (tmpArray.Length > 0)
        {
            foreach (Character enemy in tmpArray)
            {
                m_Enemies.Add(enemy);
            }
        }

        m_InGame = true;
        if (Fade.Instance.Opaque)
        {
            yield return new WaitForSeconds(0.5f);
            Fade.Instance.Out(0.5f);
            yield return new WaitForSeconds(0.5f);
        }

        if (!m_AudioSource.isPlaying && CanPlayMusic) StartCoroutine("C_MusicBegin");
    }

    // END
    private IEnumerator C_InitializeEnd()
    {
        m_AudioSource.Stop();
        CanPlayMusic = false;

        // Wait for loading
        while (Application.isLoadingLevel)
        {
            yield return 0;
        }

        // Get the level
        m_CurrentLevel = (FindObjectOfType(typeof(Level)) as Level);
        m_CurrentLevel.SetStoryProgress(0);

        // Get hero
        GameObject go = GameObject.Find("Hero");
        if (go != null)
        {
            m_Hero = go.GetComponent<Hero>();
            m_HeroTransform = m_Hero.m_CharacterModel;
        }

        if(m_Enemies != null) m_Enemies.Clear();

        m_InGame = true;
        if (Fade.Instance.Opaque)
        {
            yield return new WaitForSeconds(0.5f);
            Fade.Instance.Out(0.5f);
            yield return new WaitForSeconds(0.5f);
        }
    }
    #endregion

    #region Update
    void Update()
    {
        if (m_InGame)
        {
            m_CurrentLevel.UpdateLevel();
            if(m_Hero != null) m_Hero.UpdateCharacter();
            if (m_Enemies != null && m_Enemies.Count > 0)
            {
                foreach (Character enemy in m_Enemies)
                {
                    enemy.UpdateCharacter();
                }
            }

            if (MusicEnd())
            {
                StartCoroutine("C_MusicBegin");
            }
        }
        else if(m_Menu != null)
        {
            // Menu
            MenuResult(m_Menu.UpdateMenu());
        }
    }
    #endregion

    #region Menu
    private void MenuResult(int res)
    {
        if (res < 0)
            return;

        switch (res)
        {
                // Start new game
            case 0:
                StartGame(true);
                break;

                // Continue the game
            case 1:
                StartGame(false);
                break;

                // Quit the game
            case 2:
                Application.Quit();
                break;

                // Go back to game
            case 3:
                PauseGame(false);
                break;

                // Go back to main menu
            case 4:
                m_Menu = null;
                m_SaveManager.Save();
                StartCoroutine("C_GoBackToMainMenu");
                break;
        }
    }

    private IEnumerator C_GoBackToMainMenu()
    {
        m_AudioSource.Stop();
        Fade.Instance.In(0.5f);
        yield return new WaitForSeconds(0.5f);
        Application.LoadLevel("MainMenu");
        StartCoroutine("C_InitializeMainMenu");
    }
    #endregion

    #region Events
    public void StartGame(bool newGame)
    {
        if (newGame)
        {
            // Create a new game (delete the last one)
            m_SaveManager.CreateGame();
        }

        if (m_ProgressData.GameFinished())
        {
            Application.LoadLevel("End");
            StartCoroutine("C_InitializeEnd");
        }
        else
        {
            Application.LoadLevel(m_ProgressData.CurrentLevelName.ToString());
            StartCoroutine("C_InitializeLevel");
        }
    }

    public void PauseGame(bool pause)
    {
        if (pause)
        {
            m_InGame = false;
            GameObject go = Instantiate(Resources.Load("Prefabs/PauseMenu")) as GameObject;
            m_Menu = go.GetComponent<Menu>();
            m_Menu.Initialize();
        }
        else
        {
            m_Menu = null;
            m_InGame = true;
        }
    }

    public void NewCheckPoint(CheckPoint cp)
    {
        if (cp == m_ProgressData.CurrentCheckPoint)
        {
            //Debug.Log("Actually it's the same checkpoint...");
            return;
        }

        m_ProgressData.NewCheckpoint(cp);
        m_SaveManager.Save();
    }

    public void NextLevel()
    {
        m_InGame = false;

        if (Application.loadedLevelName == "End")
        {
            StartCoroutine("C_GoBackToMainMenu");
        }
        else
        {
            m_ProgressData.LevelEnd();
            m_SaveManager.Save();
            m_ProgressData.NextLevel();
            StartCoroutine("C_NextLevel");
        }
    }

    private IEnumerator C_NextLevel()
    {
        Fade.Instance.In(0.5f);
        yield return new WaitForSeconds(0.5f);

        // Check if this is the last one
        if ((int)m_ProgressData.CurrentLevelName == Enum.GetNames(typeof(LevelSettings.LevelName)).Length)
        {
            Debug.Log("Game finished");
            Application.LoadLevel("End");
            StartCoroutine("C_InitializeEnd");
        }
        else
        {
            Application.LoadLevel(m_ProgressData.CurrentLevelName.ToString());
            StartCoroutine("C_InitializeLevel");
        }
    }

    public void EarnPoints(int pts)
    {
        m_ProgressData.UpdateScore(pts);
    }

    public void Lost(int malus)
    {
        m_InGame = false;
        m_Hero.CharacterCanMove(false);
        m_CurrentLevel.Lost();
        StartCoroutine(C_Lost(malus));
    }

    private IEnumerator C_Lost(int malus)
    {
        yield return new WaitForSeconds(0.5f);
        // Fade in
        Fade.Instance.In(1f);
        yield return new WaitForSeconds(1f);
        // The hero come back to the last checkpoint
        m_Hero.Spawn(m_ProgressData.CurrentCheckPoint.Position);
        m_CurrentLevel.SetStoryProgress(m_ProgressData.LastCheckPoint.StoryStep);
        m_CurrentLevel.LoadProgress();

        yield return new WaitForSeconds(0.5f);
        m_InGame = true;
        yield return new WaitForSeconds(0.5f);
        // Fade out
        Fade.Instance.Out(1f);
        m_Hero.CharacterCanMove(true);

        // Update the score
        m_ProgressData.BackToRegisteredScore();
        m_ProgressData.UpdateRegisteredScore(malus);
    }
    #endregion

    #region Sounds
    // Hack for the music begin (smooth the (bad) beginning)
    private bool MusicEnd()
    {
        return (CanPlayMusic && !m_AudioSource.isPlaying);
    }

    private IEnumerator C_MusicBegin()
    {
        m_AudioSource.clip = m_Music;
        m_AudioSource.Play();
        m_AudioSource.volume = 0f;
        Timer timer = new Timer(5f);
        timer.Start();
        while (!timer.UpdateTime())
        {
            m_AudioSource.volume = timer.GetHermiteProgress();
            yield return 0;
        }
    }
    #endregion
}