﻿using UnityEngine;
using System.Collections;

public class Elevator : Element
{
    public Vector3 m_Movement;
    public float m_Size;
    public float m_Duration;
    public bool m_CanMove;
    public Transform m_PositionToMove;
    public float m_MoveDuration;

    public Transform m_Model;
    public InteractTrigger m_FirstTrigger;
    public InteractTrigger m_SecondTrigger;
    public Transform m_FirstFinalPos;
    public Transform m_SecondFinalPos;

    private bool m_OnMove;
    private bool m_Rise;
    private bool m_Activated;
    private Vector3 m_InitialPosition;
    private Vector3 m_FinalPosition;
    private Vector3 m_DeltaMove;
    public Vector3 DeltaMove
    {
        get { return m_DeltaMove; }
    }
    private Timer m_Timer;

    private Vector3 m_RegisteredPosition;
    private bool m_RegisteredRise;
    private bool m_RegisteredActivated;

    protected override void OnEnable()
    {
        base.OnEnable();
    }

    protected override void Start()
    {
        base.Start();
        m_Timer = new Timer(m_Duration);
        m_OnMove = false;
        m_Rise = true;
        m_Activated = false;

        SaveStatus();
    }

    public override bool UpdateElement()
    {
        // Player was not in it
        if(!m_OnMove && !m_Activated)
        {
            if (m_Rise)
            {
                // Player is now in it
                if (m_FirstTrigger.IsIn)
                {
                    m_InitialPosition = m_Model.position;
                    m_FinalPosition = m_Model.position + (m_Movement * m_Size);
                    m_Activated = true;
                    m_FirstTrigger.InteractCharacter.CharacterCanMove(false);
                    StartCoroutine(C_Move(true));
                }
                else if (m_SecondTrigger.IsIn)
                {
                    m_InitialPosition = m_Model.position;
                    m_FinalPosition = m_Model.position + (m_Movement * m_Size);
                    m_Activated = true;
                    m_SecondTrigger.InteractCharacter.CharacterCanMove(false);
                    StartCoroutine(C_Move(false));
                }
            }
            else
            {
                if (m_SecondTrigger.IsIn)
                {
                    m_InitialPosition = m_Model.position;
                    m_FinalPosition = m_Model.position - (m_Movement * m_Size);
                    m_Activated = true;
                    m_SecondTrigger.InteractCharacter.CharacterCanMove(false);
                    StartCoroutine(C_Move(true));
                }
                else if (m_FirstTrigger.IsIn)
                {
                    m_InitialPosition = m_Model.position;
                    m_FinalPosition = m_Model.position - (m_Movement * m_Size);
                    m_Activated = true;
                    m_FirstTrigger.InteractCharacter.CharacterCanMove(false);
                    StartCoroutine(C_Move(false));
                }
            }
        }

        return false;
    }

    private IEnumerator C_Move(bool rightPlace)
    {
        m_OnMove = true;

        if (!rightPlace)
        {
            // Set the model in the right place
            m_Timer.Restart();
            while (!m_Timer.UpdateTime())
            {
                m_DeltaMove = m_Model.position;
                m_Model.position = Vector3.Lerp(m_InitialPosition, m_FinalPosition, m_Timer.GetHermiteProgress());
                m_DeltaMove = m_Model.position - m_DeltaMove;
                yield return 0;
            }

            m_Rise = !m_Rise;
            m_Model.position = m_FinalPosition;

            if (m_Rise)
            {
                m_InitialPosition = m_Model.position;
                m_FinalPosition = m_Model.position + (m_Movement * m_Size);
            }
            else
            {
                m_InitialPosition = m_Model.position;
                m_FinalPosition = m_Model.position - (m_Movement * m_Size);
            }
        }

        // The player go on the elevator
        if(m_Rise) m_FirstTrigger.InteractCharacter.MustGoSomewhere(m_PositionToMove.position, m_MoveDuration);
        else m_SecondTrigger.InteractCharacter.MustGoSomewhere(m_PositionToMove.position, m_MoveDuration);

        // Wait for movement
        float timer = 0f;
        while (timer < m_MoveDuration)
        {
            timer += Time.deltaTime;
            yield return 0;
        }

        // The player is on the elevator
        if (m_Rise) m_FirstTrigger.InteractCharacter.CharacterOnPlatform(true, m_CanMove, this);
        else m_SecondTrigger.InteractCharacter.CharacterOnPlatform(true, m_CanMove, this);
        
        // The elevator move
        m_Timer.Restart();
        while (!m_Timer.UpdateTime())
        {
            m_DeltaMove = m_Model.position;
            m_Model.position = Vector3.Lerp(m_InitialPosition, m_FinalPosition, m_Timer.GetHermiteProgress());
            m_DeltaMove = m_Model.position - m_DeltaMove;
            yield return 0;
        }
        m_Model.position = m_FinalPosition;

        // The player is no more int the elevator
        if (m_Rise) m_FirstTrigger.InteractCharacter.CharacterOnPlatform(false, false, null);
        else m_SecondTrigger.InteractCharacter.CharacterOnPlatform(false, false, null);

        // The player moves out of the elevator
        if(m_Rise) m_FirstTrigger.InteractCharacter.MustGoSomewhere(m_FirstFinalPos.position, m_MoveDuration);
        else m_SecondTrigger.InteractCharacter.MustGoSomewhere(m_SecondFinalPos.position, m_MoveDuration);

        // Wait for movement
        timer = 0f;
        while (timer < m_MoveDuration)
        {
            timer += Time.deltaTime;
            yield return 0;
        }

        if (m_Rise) m_FirstTrigger.InteractCharacter.CharacterCanMove(true);
        else m_SecondTrigger.InteractCharacter.CharacterCanMove(true);

        m_Rise = !m_Rise;
        m_OnMove = false;
        m_Activated = false;
    }

    public override int SaveStatus()
    {
        m_RegisteredPosition = m_Model.position;
        m_RegisteredRise = m_Rise;
        m_RegisteredActivated = m_Activated;
        return base.SaveStatus();
    }

    public override void ReInit()
    {
        m_Model.position = m_RegisteredPosition;
        m_Rise = m_RegisteredRise;
        m_Activated = m_RegisteredActivated;
    }
}