using UnityEngine;
using System.Collections;

public class Character_Motor : MonoBehaviour
{
	// Character
	protected Transform m_CharacterModel;
	
	// Animation manager
	protected Character_Animator m_CharacterAnimator;
	
	// Physic manager
	protected Character_Physic m_CharacterPhysic;
	
	// Movement
	protected Vector3 m_MoveVectorInCam;
	protected Vector3 m_MoveVector;
	public Vector3 MoveVector
	{
		get { return m_MoveVector; }
		set { m_MoveVector = value; }
	}
	// Walk/Run
	protected float m_WalkSpeed;
	public float Speed
	{
		get { return m_WalkSpeed; }
	}
	protected float m_RunSpeed;
    protected bool m_Run;
    // Jump
    protected float m_JumpHeight;
	protected float m_JumpSpeed;
	protected float m_FallSpeed;
    protected bool m_Jump;
	// Vertical velocity
	public float m_MaxVelocity = 0.5f;
	protected float m_VerticalVelocity;
	public float VerticalVelocity
	{
		get { return m_VerticalVelocity; }
		set { m_VerticalVelocity = value; }
	}
    // On movable platforms
    protected bool m_IsOnPlatform;
    protected bool m_CanMoveOnPlatform;
    protected Elevator m_Platform;
    // Must move somewhere
    private bool m_MoveSomewhere;
    private Vector3 m_MoveInitialPos;
    private Vector3 m_MoveFinalPos;
    private Timer m_MoveTimer;
    // Grounded
	public virtual bool isGrounded
	{
		get { return true; }
	}
	// Initial vector null -> for ground tests by the unity Character controller
	protected Vector3 m_VectorNull;
	public Vector3 VectorNull
	{
		get { return m_VectorNull; }
	}

    //public void Character_Motor(Transform character, CharacterController controller, Character_Animator animator, Character_Physic physic)
	public virtual void Initialize(Transform character, Character_Animator animator, Character_Physic physic)
	{
		m_CharacterModel = character;
		m_CharacterAnimator = animator;
		m_CharacterPhysic = physic;
		m_MoveVector = Vector3.zero;
		m_Run = false;
		m_WalkSpeed = 3f;
		m_RunSpeed = 7f;
        m_JumpHeight = 1.2f;
		m_JumpSpeed = 0.4f;
        m_Jump = false;
		m_FallSpeed = 0.5f;
		m_VerticalVelocity = 0f;
		m_VectorNull = new Vector3(0f,-0.01f,0f);
        m_IsOnPlatform = false;
        m_CanMoveOnPlatform = false;
        m_MoveSomewhere = false;
        m_MoveTimer = new Timer();
	}
	
	#region Update
	public virtual void UpdateMotor()
	{
		ProcessMotion();
	}
	
	protected virtual void ProcessMotion()
	{
        if (m_MoveSomewhere)
        {
            if (!m_MoveTimer.UpdateTime())
            {
                m_MoveVector = Vector3.Lerp(m_MoveInitialPos, m_MoveFinalPos, m_MoveTimer.GetHermiteProgress()) - m_CharacterModel.position;
            }
            else
            {
                m_MoveVector = m_MoveFinalPos - m_CharacterModel.position;
                m_MoveSomewhere = false;
            }
        }

        m_CharacterAnimator.SetSpeed(Mathf.Abs(m_MoveVector.x));

		// Look for character orientation
		CharacterOrientation();
		
		// Normalize MoveVector if its magnitude > 1
		if(m_MoveVector.magnitude > 1)
		{
			m_MoveVector = Vector3.Normalize(m_MoveVector);
		}

        if (!m_MoveSomewhere)
        {
            if (!m_Run) m_MoveVector *= m_WalkSpeed * Time.deltaTime;
            else m_MoveVector *= m_RunSpeed * Time.deltaTime;
        }
		
		ApplyGravity();
		
		ApplyMovement();
	}
	
	// Orientation
	protected void CharacterOrientation()
	{
		Vector3 target = m_MoveVector;
		target.y = 0f;
		target += m_CharacterModel.position;
		m_CharacterModel.LookAt(target);
	}
	
	// Gravity
	protected virtual void ApplyGravity()
	{
	}
	
	protected virtual void ApplyMovement()
	{
        m_CharacterModel.position += m_MoveVector;
	}
	
	// Save the velocity
	public void ReinitMoveVector()
	{
		m_VerticalVelocity = m_MoveVector.y;
		m_MoveVector = Vector3.zero;
	}
	#endregion
	
	#region Tests
	#endregion

    #region State events
    // Called when the player lost
    public void ReInit()
    {
        m_IsOnPlatform = false;
        m_MoveSomewhere = false;
    }

    // The character is on a platform (and must follow it)
    public virtual void IsOnPlatform(bool isOn, bool canMove, Elevator e)
    {
        m_IsOnPlatform = isOn;
        m_CanMoveOnPlatform = canMove;
        if (m_IsOnPlatform) m_Platform = e;
    }

    // The character must go in a defined position
    public virtual void MustGoSomewhere(Vector3 position, float duration)
    {
        m_MoveSomewhere = true;
        m_MoveInitialPos = m_CharacterModel.position;
        m_MoveFinalPos = position;
        m_MoveTimer.Restart(duration);
    }
    #endregion

    #region Inputs events
    public virtual void Jump()
    {
        // TODO
    }

	public void Run(bool run)
	{
		m_Run = run;
	}
	#endregion
}