﻿using UnityEngine;
using System.Collections;

public class Door : Element
{
    private Diode[] m_Diodes;
    private bool[] m_RegisteredDiodesOn;
    private bool[] m_DiodesOn;
    public int m_NbDiodes;

    public Animator m_Animator;
    private bool m_DoorOpened;

    public bool m_OpenAtContact = false;
    private InteractTrigger m_Trigger;

    protected override void OnEnable()
    {
        base.OnEnable();
        m_Diodes = GetComponentsInChildren<Diode>();

        // If the door is opened when the player is near
        if(m_OpenAtContact)
        {
            //Debug.Log("Contact");
            m_Trigger = GetComponent<InteractTrigger>();
        }
        // Else it is locked
        else
        {
            if(m_Diodes.Length > 0)
            {
                m_DiodesOn = new bool[m_NbDiodes];
                m_RegisteredDiodesOn = new bool[m_NbDiodes];
                for (int i = 0; i < m_DiodesOn.Length; ++i)
                {
                    m_DiodesOn[i] = false;
                    m_RegisteredDiodesOn[i] = false;
                }
                m_DoorOpened = false;
            }
        }
    }

    protected override void Start()
    {
        base.Start();
    }

    public override bool UpdateElement()
    {
        // If the door is opened when the player is near => check the trigger
        if (m_OpenAtContact)
        {
            if (!m_DoorOpened && m_Trigger.IsIn)
            {
                Open();
            }
            else if (m_DoorOpened && !m_Trigger.IsIn)
            {
                Close();
            }
        }

        return base.UpdateElement();
    }

    public override int SaveStatus()
    {
        if (m_NbDiodes > 0)
        {
            for (int i = 0; i < m_DiodesOn.Length; ++i)
            {
                m_RegisteredDiodesOn[i] = m_DiodesOn[i];
            }
        }
        return base.SaveStatus();
    }

    public override void ReInit()
    {
        base.ReInit();
        if (m_OpenAtContact || m_NbDiodes == 0)
            return;

        for (int i = 0; i < m_DiodesOn.Length; ++i)
        {
            m_DiodesOn[i] = m_RegisteredDiodesOn[i];
        }

        bool open = true;
        for (int i = 0; i < m_DiodesOn.Length; ++i)
        {
            if (!m_DiodesOn[i])
            {
                foreach (Diode d in m_Diodes)
                {
                    d.SetOn(i+1, false);
                    open = false;
                }
            }
        }

        if (m_DoorOpened && !open)
        {
            Close();
        }
    }

    public bool SwitchDiodesOn(int iterator)
    {
        for (int i = 0; i < m_Diodes.Length; ++i)
        {
            m_Diodes[i].SetOn(iterator, true);
        }

        m_DiodesOn[iterator-1] = true;

        bool result = true;
        foreach (bool res in m_DiodesOn)
        {
            if (!res) result = false;
        }

        if (result)
        {
            Open();
        }

        return result;
    }

    public void Open()
    {
        m_Animator.SetBool("Open", true);
        m_DoorOpened = true;
    }

    public void Close()
    {
        m_Animator.SetBool("Open", false);
        m_DoorOpened = false;
    }

    public bool DiodeAlreadySwitchOn(int iterator)
    {
        //Debug.Log("Check for diode " + iterator + " : " + m_RegisteredDiodesOn[iterator]);
        return m_RegisteredDiodesOn[iterator];
    }
}