﻿using UnityEngine;
using System.Collections;

public class ComputerScreen : MonoBehaviour
{
    public enum Anchor
    {
        UpperLeft,
        UpperCenter,
        UpperRight,
        Left,
        Center,
        Right,
        DownLeft,
        DownCenter,
        DownRight
    };

    public Texture2D m_Texture;
    public Anchor m_Anchor;
    public Vector2 m_Offset;
    public bool m_FullWidth;
    public bool m_FullHeight;
    public bool m_KeepRatio;
    public Vector2 m_Size;
    public float m_PlanUnitSize = 10f;

    void OnEnable()
    {
        // Apply texture
        renderer.material.SetTexture("_MainTex", m_Texture);

        float distance = Mathf.Abs(Camera.main.transform.position.z - transform.position.z);
        float height, width, ratio;

        // Set size
        // -> FullScreen
        if (m_FullHeight && m_FullWidth)
        {
            // => Ignore anchor and size
            height = Mathf.Tan(Camera.main.fieldOfView * 0.5f * Mathf.Deg2Rad) * distance / m_PlanUnitSize;
            height *= 2f;

            // Apply original texture or screen ratio
            if (m_KeepRatio) ratio = m_Texture.width / m_Texture.height;
            else ratio = (float)Screen.width / Screen.height;

            // Set the correct size for the plane
            transform.localScale = new Vector3(height * ratio, 1f, height);

            // Set position (center anyway)
            transform.position = new Vector3(Camera.main.transform.position.x, Camera.main.transform.position.y, transform.position.z);
        }
        // -> Full height
        else if(m_FullHeight)
        {
            // Calculate the height
            height = Mathf.Tan(Camera.main.fieldOfView * 0.5f * Mathf.Deg2Rad) * distance / m_PlanUnitSize;
            height *= 2f;

            // Apply original texture or size ratio
            if (m_KeepRatio) ratio = m_Texture.width / m_Texture.height;
            else ratio = m_Size.x / m_Size.y;

            // Set the correct size for the plane
            transform.localScale = new Vector3(height * ratio, 1f, height);

            // Set position (ignore upper or down, vertically centered anyway)
            switch (m_Anchor)
            {
            case Anchor.UpperLeft: case Anchor.Left: case Anchor.DownLeft:
                ratio = (float)Screen.width / Screen.height;
                transform.position = new Vector3(Camera.main.transform.position.x - height * ratio * 0.5f * m_PlanUnitSize + transform.localScale.x * 0.5f * m_PlanUnitSize, Camera.main.transform.position.y, transform.position.z);
                break;

            case Anchor.UpperCenter: case Anchor.Center: case Anchor.DownCenter:
                transform.position = new Vector3(Camera.main.transform.position.x, Camera.main.transform.position.y, transform.position.z);
                break;

            case Anchor.UpperRight: case Anchor.Right: case Anchor.DownRight:
                ratio = (float)Screen.width / Screen.height;
                transform.position = new Vector3(Camera.main.transform.position.x + height * ratio * 0.5f * m_PlanUnitSize - transform.localScale.x * 0.5f * m_PlanUnitSize, Camera.main.transform.position.y, transform.position.z);
                break;
            }
        }
        else if (m_FullWidth)
        {
            // Calculate the height
            height = Mathf.Tan(Camera.main.fieldOfView * 0.5f * Mathf.Deg2Rad) * distance / m_PlanUnitSize;
            height *= 2f;

            // Calculate ratio to have the plane weight
            ratio = (float)Screen.width / Screen.height;
            width = height * ratio;

            // Apply original texture or size ratio
            if (m_KeepRatio) ratio = m_Texture.height / m_Texture.width;
            else ratio = m_Size.y / m_Size.x;

            // Set the correct size for the plane
            transform.localScale = new Vector3(width, 1f, width * ratio);

            // Set position (ignore left or right, horizontally centered anyway)
            switch (m_Anchor)
            {
            case Anchor.UpperLeft: case Anchor.UpperCenter: case Anchor.UpperRight:
                transform.position = new Vector3(Camera.main.transform.position.x, Camera.main.transform.position.y + height * 0.5f * m_PlanUnitSize - transform.localScale.z * 0.5f * m_PlanUnitSize, transform.position.z);
                break;

            case Anchor.Left: case Anchor.Center: case Anchor.Right:
                transform.position = new Vector3(Camera.main.transform.position.x, Camera.main.transform.position.y, transform.position.z);
                break;

            case Anchor.DownLeft: case Anchor.DownCenter: case Anchor.DownRight:
                ratio = (float)Screen.width / Screen.height;
                transform.position = new Vector3(Camera.main.transform.position.x, Camera.main.transform.position.y - height * 0.5f * m_PlanUnitSize + transform.localScale.z * 0.5f * m_PlanUnitSize, transform.position.z);
                break;
            }
        }
        else
        {
            width = m_Size.x;
            if (m_KeepRatio) height = width * m_Texture.height / m_Texture.width;
            else height = m_Size.y;

            // Set the correct size for the plane
            transform.localScale = new Vector3(width, 1f, height);

            // Width and height unit of screen
            height = Mathf.Tan(Camera.main.fieldOfView * 0.5f * Mathf.Deg2Rad) * distance / m_PlanUnitSize;
            height *= 2f;
            ratio = (float)Screen.width / Screen.height;
            width = height * ratio;
            // Set Position
            switch (m_Anchor)
            {
                case Anchor.UpperLeft:
                    transform.position = new Vector3(Camera.main.transform.position.x - width * 0.5f * m_PlanUnitSize + transform.localScale.x * 0.5f * m_PlanUnitSize, Camera.main.transform.position.y + height * 0.5f * m_PlanUnitSize - transform.localScale.z * 0.5f * m_PlanUnitSize, transform.position.z);
                    break;

                case Anchor.UpperCenter:
                    transform.position = new Vector3(Camera.main.transform.position.x, Camera.main.transform.position.y + height * 0.5f * m_PlanUnitSize - transform.localScale.z * 0.5f * m_PlanUnitSize, transform.position.z);
                    break;

                case Anchor.UpperRight:
                    transform.position = new Vector3(Camera.main.transform.position.x + width * 0.5f * m_PlanUnitSize - transform.localScale.x * 0.5f * m_PlanUnitSize, Camera.main.transform.position.y + height * 0.5f * m_PlanUnitSize - transform.localScale.z * 0.5f * m_PlanUnitSize, transform.position.z);
                    break;

                case Anchor.Left:
                    transform.position = new Vector3(Camera.main.transform.position.x - width * 0.5f * m_PlanUnitSize + transform.localScale.x * 0.5f * m_PlanUnitSize, Camera.main.transform.position.y, transform.position.z);
                    break;

                case Anchor.Center:
                    transform.position = new Vector3(Camera.main.transform.position.x, Camera.main.transform.position.y, transform.position.z);
                    break;

                case Anchor.Right:
                    transform.position = new Vector3(Camera.main.transform.position.x + width * 0.5f * m_PlanUnitSize - transform.localScale.x * 0.5f * m_PlanUnitSize, Camera.main.transform.position.y, transform.position.z);
                    break;

                case Anchor.DownLeft:
                    transform.position = new Vector3(Camera.main.transform.position.x - width * 0.5f * m_PlanUnitSize + transform.localScale.x * 0.5f * m_PlanUnitSize, Camera.main.transform.position.y - height * 0.5f * m_PlanUnitSize + transform.localScale.z * 0.5f * m_PlanUnitSize, transform.position.z);
                    break;

                case Anchor.DownCenter:
                    transform.position = new Vector3(Camera.main.transform.position.x, Camera.main.transform.position.y - height * 0.5f * m_PlanUnitSize + transform.localScale.z * 0.5f * m_PlanUnitSize, transform.position.z);
                    break;

                case Anchor.DownRight:
                    transform.position = new Vector3(Camera.main.transform.position.x + width * 0.5f * m_PlanUnitSize - transform.localScale.x * 0.5f * m_PlanUnitSize, Camera.main.transform.position.y - height * 0.5f * m_PlanUnitSize + transform.localScale.z * 0.5f * m_PlanUnitSize, transform.position.z);
                    break;
            }
        }
    }
}