using System.Collections.Generic;
using UnityEngine;
using UnityEngine.Serialization;

namespace Global
{
    public class PlayerStatsManager : MonoBehaviour
    {
        public static PlayerStatsManager instance;

        [FormerlySerializedAs("_stars")] [SerializeField]
        private int stars;

        [SerializeField] private SkinScheme[] availableSkins;
        private List<SkinScheme> _ownedSkins;
        private SkinScheme _currentSkin;
        [SerializeField] private LevelSet levels;
        private bool[][] LevelsStarsCollected { get; set; }
        private int[] _levelsBestScores;
        public LevelComposition currentLevelToPlay;

        private void Awake()
        {
            if (instance == null)
            {
                instance = this;
                DontDestroyOnLoad(gameObject);
            }
            else
            {
                Destroy(gameObject);
            }
            
            // if there is saved data, load it
            // else initialize with default values
            if (PlayerPrefs.HasKey("Stars"))
            {
                stars = PlayerPrefs.GetInt("Stars");
                // load owned skins
                string ownedSkinsString = PlayerPrefs.GetString("OwnedSkins");
                string[] ownedSkinsNames = ownedSkinsString.Split(',');
                foreach (string skinName in ownedSkinsNames)
                {
                    foreach (SkinScheme skin in availableSkins)
                    {
                        if (skin.skinName == skinName)
                        {
                            if (_ownedSkins == null)
                            {
                                _ownedSkins = new List<SkinScheme>();
                            }
                            _ownedSkins.Add(skin);
                            break;
                        } else if (skinName == "Default")
                        {
                            if (_ownedSkins == null)
                            {
                                _ownedSkins = new List<SkinScheme>();
                            }
                            // create a default skin and add it to owned skins with color and name "Default"
                            SkinScheme defaultSkin = ScriptableObject.CreateInstance<SkinScheme>();
                            defaultSkin.skinName = "Default";
                            _ownedSkins.Add(defaultSkin);
                            _ownedSkins[0].skinStartColor = new Color(255, 255, 255, 255);
                            _ownedSkins[0].skinEndColor = new Color(255, 255, 255, 0);
                            _ownedSkins[0].trailWidth = 0.05f;
                            
                            break;
                        }
                    }
                }
                // load current skin
                string currentSkinName = PlayerPrefs.GetString("CurrentSkin");
                foreach (SkinScheme skin in _ownedSkins)
                {
                    if (skin.skinName == currentSkinName)
                    {
                        _currentSkin = skin;
                        break;
                    }
                }
                // load levels stars collected
                LevelsStarsCollected = new bool[levels.levels.Count][];
                for (int i = 0; i < levels.levels.Count; i++)
                {
                    string starsString = PlayerPrefs.GetString("LevelStarsCollected_" + i);
                    LevelsStarsCollected[i] = new bool[3];
                    for (int j = 0; j < 3; j++)
                    {
                        LevelsStarsCollected[i][j] = starsString.Length > j && starsString[j] == '1';
                    }
                }
                // load levels best scores
                _levelsBestScores = new int[levels.levels.Count];
                for (int i = 0; i < levels.levels.Count; i++)
                {
                    _levelsBestScores[i] = PlayerPrefs.GetInt("LevelBestScore_" + i);
                }
            }
            else
            {
                InitializeDefaultStats();
                SaveStats();
            }
        }

        private void InitializeDefaultStats()
        {
            _ownedSkins = new List<SkinScheme>();
            // create a default skin and add it to owned skins with color and name "Default"
            _ownedSkins.Add(ScriptableObject.CreateInstance<SkinScheme>());
            _ownedSkins[0].skinName = "Default";
            _ownedSkins[0].skinStartColor = new Color(255, 255, 255, 255);
            _ownedSkins[0].skinEndColor = new Color(255, 255, 255, 0);
            _ownedSkins[0].trailWidth = 0.05f;
            _currentSkin = _ownedSkins[0];
            // initialize levelsStarsCollected with false values
            LevelsStarsCollected = new bool[levels.levels.Count][];
            // 3 stars per level
            for (int i = 0; i < levels.levels.Count; i++)
            {
                LevelsStarsCollected[i] = new bool[3];
                for (int j = 0; j < 3; j++)
                {
                    LevelsStarsCollected[i][j] = false;
                }
            }

            _levelsBestScores = new int[levels.levels.Count];
            // DEBUG
            /*LevelsStarsCollected[0][0] = true;
            LevelsStarsCollected[1][0] = true;
            LevelsStarsCollected[2][0] = true;
            LevelsStarsCollected[3][0] = true;
            LevelsStarsCollected[4][0] = true;
            LevelsStarsCollected[5][0] = true;
            LevelsStarsCollected[6][0] = true;
            LevelsStarsCollected[7][0] = true;
            LevelsStarsCollected[8][0] = true;*/
            //levelsStarsCollected[0][1] = true;
            //levelsStarsCollected[0][2] = true;
            //_levelsBestScores[0] = 1500;
        }

        public int GetStars()
        {
            return stars;
        }

        public void SetStars(int newStars)
        {
            this.stars = newStars;
            SaveStats();
        }

        public SkinScheme[] GetAvailableSkins()
        {
            return availableSkins;
        }

        public SkinScheme[] GetOwnedSkins()
        {
            return _ownedSkins.ToArray();
        }

        public void AddOwnedSkin(string skinName)
        {
            if (!_ownedSkins.Exists(skin => skin.skinName == skinName))
            {
                foreach (SkinScheme skin in availableSkins)
                {
                    if (skin.skinName == skinName)
                    {
                        _ownedSkins.Add(skin);
                        break;
                    }
                }
            }
            SaveStats();
        }

        public void SetCurrentSkin(string skinName)
        {
            foreach (SkinScheme skin in _ownedSkins)
            {
                if (skin.skinName == skinName)
                {
                    _currentSkin = skin;
                    break;
                }
            }
            SaveStats();
        }

        public SkinScheme GetCurrentSkin()
        {
            return _currentSkin;
        }

        public bool[] GetLevelStarsCollected(int levelIndex)
        {
            return LevelsStarsCollected[levelIndex];
        }

        public string GetLevelScoreToReach(int levelIndex, int starsCount)
        {

            switch (starsCount)
            {
                case 0:
                    return levels.levels[levelIndex].scoreForStar1.ToString();
                case 1:
                    return levels.levels[levelIndex].scoreForStar2.ToString();
                case 2:
                    return levels.levels[levelIndex].scoreForStar3.ToString();
                default:
                    return "0";
            }
        }

        public string GetLevelBestScore(int levelIndex)
        {
            return _levelsBestScores[levelIndex].ToString();
        }

        public LevelComposition GetLevelByIndex(int levelIndex)
        {
            return levels.levels[levelIndex];
        }

        public int GetCurrentLevelIndex()
        {
            for (int i = 0; i < levels.levels.Count; i++)
            {
                if (levels.levels[i] == currentLevelToPlay)
                {
                    return i;
                }
            }

            return -1;
        }

        public void UpdateLevelBestScore(int currentLevelIndex, int finalScore)
        {
            _levelsBestScores[currentLevelIndex] = finalScore;
            SaveStats();
        }

        public void UpdateLevelStarsCollected(int currentLevelIndex, int starsEarned)
        {
            for (int i = 0; i < starsEarned; i++)
            {
                LevelsStarsCollected[currentLevelIndex][i] = true;
            }
            SaveStats();
        }

        private void SaveStats()
        {
            PlayerPrefs.SetInt("Stars", stars);
            PlayerPrefs.SetString("CurrentSkin", _currentSkin.skinName);
            // Save owned skins as a comma-separated string
            string ownedSkinsString = string.Join(",", _ownedSkins.ConvertAll(skin => skin.skinName).ToArray());
            PlayerPrefs.SetString("OwnedSkins", ownedSkinsString);
            // Save levels stars collected
            for (int i = 0; i < LevelsStarsCollected.Length; i++)
            {
                string starsString = "";
                for (int j = 0; j < LevelsStarsCollected[i].Length; j++)
                {
                    starsString += LevelsStarsCollected[i][j] ? "1" : "0";
                }

                PlayerPrefs.SetString("LevelStarsCollected_" + i, starsString);
            }

            // Save levels best scores
            for (int i = 0; i < _levelsBestScores.Length; i++)
            {
                PlayerPrefs.SetInt("LevelBestScore_" + i, _levelsBestScores[i]);
            }

            PlayerPrefs.Save();
        }
    }
}
