using System.Collections.Generic;
using UnityEngine;

namespace Game
{
    public class TapPatternDetector : MonoBehaviour
    {
        [Header("Tap Settings")]
        [SerializeField, Tooltip("Radius du SphereCast pour détecter les taps.")]
        private float tapRadius = 0.1f;
        [SerializeField, Tooltip("Distance max du SphereCast pour détecter les taps.")]
        private float tapMaxDistance = 15f;
        [SerializeField, Tooltip("Distance maximale de mouvement pour qu'un tap soit valide.")]
        private float maxTapMovementDistance = 0.05f;

        [Header("Debug Draw")]
        [SerializeField, Tooltip("Affiche une représentation du SphereCast dans la vue de la scène lors d'un tap.")]
        private bool drawSpherecastGizmo = true;
        [SerializeField]
        private Color gizmoColor = Color.cyan;
        [SerializeField, Tooltip("Durée d'affichage de la visualisation (en secondes).")]
        private float gizmoDuration = 0.25f;

        private CursorController _cursorController;
        private Camera _cam;

        // Etat pour la visualisation du dernier SphereCast
        private Ray _lastRay;
        private float _lastTravelDistance;
        private bool _hasLastSpherecast;
        private float _gizmoExpireTime;

        // Position de départ pour vérifier le mouvement
        private Vector3 _startPosition;

        private void Awake()
        {
            _cursorController = GetComponent<CursorController>();
        }

        private void Start()
        {
            _cam = _cursorController.GetCamera();
        }

        public void OnBeginPress(Vector3 startPosition)
        {
            _startPosition = startPosition;
        }

        public void PerformTap(Vector2 screenPosition, Vector3 currentPosition)
        {
            // Vérifier si le mouvement était minimal (c'est un vrai tap)
            float movementDistance = Vector3.Distance(_startPosition, currentPosition);
            if (movementDistance > maxTapMovementDistance)
            {
                // Trop de mouvement, ce n'est pas un tap
                return;
            }

            if (_cam == null)
            {
                Debug.LogError("La caméra n'est pas assignée dans TapDetector.");
                return;
            }
        
            // On ne fait rien si un cercle est en cours de formation/animation
            bool isCircleFormed = _cursorController.CircleFormedThisSlice;
            //Debug.Log($"Checking if circle was formed: {isCircleFormed}");
            if (isCircleFormed)
            {
                return;
            }

            // Si la caméra n'occupe pas tout l'écran, convertir la position d'écran
            Rect pr = _cam.pixelRect;
            Vector2 camPix = screenPosition - pr.min;
            Ray ray = _cam.ScreenPointToRay(new Vector3(camPix.x, camPix.y, 0f));
            // Lancer le SphereCast
            bool didHit = Physics.SphereCast(ray, tapRadius, out RaycastHit hit, tapMaxDistance);

            // --- Visualisation ---
            if (drawSpherecastGizmo)
            {
                _lastRay = ray;
                _lastTravelDistance = didHit ? hit.distance : tapMaxDistance;
                _hasLastSpherecast = true;
                _gizmoExpireTime = Time.time + gizmoDuration;

                // Trace la direction principale dans la scène
                Debug.DrawRay(ray.origin, ray.direction * _lastTravelDistance, gizmoColor, gizmoDuration);
            }

            if (didHit)
            {
                if (hit.collider.CompareTag("Enemy"))
                {
                    Debug.Log($"Enemy hit: {hit.collider.name}");
                    Enemy enemy = hit.collider.GetComponent<Enemy>();
                    if (enemy != null)
                    {
                        _cursorController.HandleEnemiesHit(new List<Enemy>() { enemy }, PatternType.Tap);
                    }
                }
            }
        }

        private void OnDrawGizmos()
        {
            if (!drawSpherecastGizmo || !_hasLastSpherecast)
                return;

            // N'afficher que pendant la fenêtre définie
            if (Application.isPlaying && Time.time > _gizmoExpireTime)
                return;

            Gizmos.color = gizmoColor;

            // Centres de la "capsule" balayée par le SphereCast
            Vector3 startCenter = _lastRay.origin;
            Vector3 endCenter = _lastRay.origin + _lastRay.direction * _lastTravelDistance;

            // Sphères aux extrémités pour représenter le rayon du spherecast
            Gizmos.DrawWireSphere(startCenter, tapRadius);
            Gizmos.DrawWireSphere(endCenter, tapRadius);

            // Lien entre les deux centres pour montrer la trajectoire
            Gizmos.DrawLine(startCenter, endCenter);
        }
    }
}
