using System;
using System.Collections;
using Global;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.Audio;
using UnityEngine.Serialization;

namespace Game
{
    public enum Side
    {
        Left,
        Right
    }


    [RequireComponent(typeof(Rigidbody))]
    public class Enemy : MonoBehaviour
    {
        public static List<Enemy> Enemies = new List<Enemy>();

        private static readonly int Dead = Animator.StringToHash("Dead");
        private Animator _animator;
        [FormerlySerializedAs("_targetWaypoint")] public Transform targetWaypoint;
        public EnemyStats enemyStats;
        private GameObject _killEffectPrefab;
        private GameObject _sweatEffectPrefab;
        public bool isDead;
        public bool isCaptured;
        public AudioResource hitSound;
        public AudioResource deathSound;
        private float _speed;
        private int _currentHealth;
        public PatternType capturePattern;
        private const float OutlineDuration = 0.8f;
        private SoundManager _soundManager;
        
        public void Awake()
        {
            Enemies.Add(this);
            _speed = enemyStats.speed;
            _currentHealth = enemyStats.health;
            transform.localScale = Vector3.one * enemyStats.size;
            _killEffectPrefab = transform.Find("Star hit")?.gameObject;
            _sweatEffectPrefab = transform.Find("Sweat hit")?.gameObject;
            if (_killEffectPrefab != null)
            {
                _killEffectPrefab.SetActive(false);
            }
            _animator = GetComponentInChildren<Animator>();
            _soundManager = SoundManager.Instance;
        }

        private void OnDestroy()
        {
            Enemies.Remove(this);
        }

        private void Update()
        {
            if (targetWaypoint == null || isDead || isCaptured) return;
            var direction = new Vector2(
                targetWaypoint.position.x - transform.position.x,
                targetWaypoint.position.z - transform.position.z
            ).normalized;
            transform.position += new Vector3(direction.x, 0, direction.y) * (_speed * Time.deltaTime);
            // rotate to face the waypoint
            var toRotation = Quaternion.LookRotation(direction, Vector3.up);
            transform.rotation = Quaternion.RotateTowards(transform.rotation, toRotation, 360 * Time.deltaTime);
        }

        public void Capture()
        {
            isCaptured = true;
        }

        public void KillEnemy()
        {
            isDead = true;
            _killEffectPrefab.SetActive(true);
            _animator.SetTrigger(Dead);
            StartCoroutine(KillAnimation());
            // Add score to player
            GameManager.instance.score += enemyStats.scoreValue;
            Destroy(gameObject, 0.6f);
        }

        private IEnumerator KillAnimation()
        {
            // increase scale for 0.2 seconds and then decrease it to 0 in 0.8 seconds
            var elapsedTime = 0f;
            var initialScale = transform.localScale;
            var enlargedScale = initialScale * 1.5f;
            while (elapsedTime < 0.2f)
            {
                transform.localScale = Vector3.Lerp(initialScale, enlargedScale, (elapsedTime / 0.2f));
                elapsedTime += Time.deltaTime;
                yield return null;
            }
            elapsedTime = 0f;
            while (elapsedTime < 0.4f)
            {
                transform.localScale = Vector3.Lerp(enlargedScale, Vector3.zero, (elapsedTime / 0.4f));
                elapsedTime += Time.deltaTime;
                yield return null;
            }
        }

        public void TakeDamage()
        {
            if (isDead) return;
            _currentHealth -= 1;
            if (_currentHealth <= 0)
            {
                PlayDeathSound();
                KillEnemy();
            }
            else
            {
                PlayHitSound();
                StartCoroutine(StunCoroutine(0.4f));
            }
        }
        
        private void PlayHitSound()
        {
            _soundManager.PlaySoundAsset(hitSound);
        }
        
        private void PlayDeathSound()
        {
            _soundManager.PlaySoundAsset(deathSound);
        }

        private IEnumerator StunCoroutine(float duration)
        {
            _speed /= 5;
            if (_sweatEffectPrefab != null)
            {
                _sweatEffectPrefab.GetComponent<ParticleSystem>().Play();
            }
            yield return new WaitForSeconds(duration);
            _speed = enemyStats.speed;
        }
        
        public IEnumerator ShowOutline()
        {
            var outline = GetComponent<Outline>();
            if (outline == null) yield break;
            outline.enabled = true;
            yield return new WaitForSeconds(OutlineDuration);
            if (outline == null) yield break;
            outline.enabled = false;
        }
    }
}