using System.Collections.Generic;
using UnityEngine;

namespace Game
{
    public class CutPatternDetector : MonoBehaviour
    {
        private CursorController _cursorController;
        private Camera _cam;
        private GameObject _detectionSphere;
        private SphereCollider _sphereCollider;
        private readonly HashSet<Enemy> _enemiesHit = new();
        
        private bool _isSlicing;
        private Vector3 _lastPosition;
        private float _currentSpeed;
        private float _totalDistanceTraveled;

        [Header("Detection Settings")]
        [SerializeField] private float projectionPlaneY = 0.15f;
        [SerializeField] private float sphereRadius = 0.5f;
        [SerializeField] private float minSpeedForCut = 1f;
        [SerializeField] private float minCutDistance = 0.5f;
        [SerializeField] private float interpolationStep = 0.05f;
        [SerializeField] private LayerMask enemyLayer;

        private void Awake()
        {
            _cursorController = GetComponent<CursorController>();
            CreateDetectionSphere();
        }

        private void Start() => _cam = _cursorController.GetCamera();

        private void CreateDetectionSphere()
        {
            _detectionSphere = new GameObject("CutDetectionSphere")
            {
                transform = { parent = transform }
            };
            
            _sphereCollider = _detectionSphere.AddComponent<SphereCollider>();
            _sphereCollider.radius = sphereRadius;
            _sphereCollider.isTrigger = true;
            
            var rb = _detectionSphere.AddComponent<Rigidbody>();
            rb.isKinematic = true;
            rb.useGravity = false;
            
            _detectionSphere.AddComponent<CutSphereCollisionDetector>().Initialize(this);
            _detectionSphere.SetActive(false);
        }

        public void OnBeginSlice()
        {
            _enemiesHit.Clear();
            _isSlicing = true;
            _detectionSphere.SetActive(true);
            _lastPosition = Vector3.zero;
            _currentSpeed = 0f;
            _totalDistanceTraveled = 0f;
        }

        public void AddPoint(Vector3 point)
        {
            if (!_isSlicing || _cam == null) return;
            
            Vector3 projectedPoint = ProjectOntoPlane(point);
            if (projectedPoint == Vector3.zero) return;
            
            UpdateMovement(projectedPoint);
            _detectionSphere.transform.position = projectedPoint;
            _sphereCollider.enabled = IsValidCutMovement();
        }

        private Vector3 ProjectOntoPlane(Vector3 point)
        {
            Vector3 screenPoint = _cam.WorldToScreenPoint(point);
            Ray ray = _cam.ScreenPointToRay(screenPoint);
            
            if (Mathf.Abs(ray.direction.y) < 0.0001f) return Vector3.zero;
            
            float t = (projectionPlaneY - ray.origin.y) / ray.direction.y;
            return t > 0 ? ray.origin + ray.direction * t : Vector3.zero;
        }

        private void UpdateMovement(Vector3 newPosition)
        {
            if (_lastPosition == Vector3.zero)
            {
                _lastPosition = newPosition;
                return;
            }
            
            float distance = Vector3.Distance(newPosition, _lastPosition);
            _totalDistanceTraveled += distance;
            _currentSpeed = distance / Time.deltaTime;
            
            if (IsValidCutMovement() && distance > 0.001f)
            {
                DetectEnemiesAlongPath(_lastPosition, newPosition);
            }
            
            _lastPosition = newPosition;
        }

        private bool IsValidCutMovement() => 
            _currentSpeed >= minSpeedForCut && _totalDistanceTraveled >= minCutDistance;

        private void DetectEnemiesAlongPath(Vector3 start, Vector3 end)
        {
            float distance = Vector3.Distance(start, end);
            int stepCount = Mathf.CeilToInt(distance / interpolationStep);
            
            if (stepCount <= 1) return;
            
            for (int i = 0; i <= stepCount; i++)
            {
                Vector3 pos = Vector3.Lerp(start, end, i / (float)stepCount);
                CheckEnemiesAtPosition(pos);
            }
        }

        private void CheckEnemiesAtPosition(Vector3 position)
        {
            Collider[] overlaps = Physics.OverlapSphere(position, sphereRadius, enemyLayer);
            foreach (var col in overlaps)
            {
                var enemy = col.GetComponent<Enemy>();
                if (enemy != null && _enemiesHit.Add(enemy))
                {
                    ProcessCutImmediate(enemy);
                }
            }
        }

        private void ProcessCutImmediate(Enemy enemy)
        {
            Debug.Log($"CutPatternDetector: Cut on {enemy.name} (speed: {_currentSpeed:F2})");
            _cursorController.HandleEnemiesHit(new List<Enemy> { enemy }, PatternType.Cut);
        }

        public void ProcessCut()
        {
            _isSlicing = false;
            _detectionSphere.SetActive(false);
            _currentSpeed = 0f;
            _totalDistanceTraveled = 0f;
            _enemiesHit.Clear();
        }
        
        public void OnEnemyEnter(Enemy enemy)
        {
            if (_isSlicing && enemy != null && IsValidCutMovement() && _enemiesHit.Add(enemy))
            {
                ProcessCutImmediate(enemy);
            }
        }
        
        #if UNITY_EDITOR
        private void OnDrawGizmos()
        {
            if (_detectionSphere == null || !_detectionSphere.activeSelf) return;
            
            Gizmos.color = IsValidCutMovement() 
                ? new Color(1, 0, 0, 0.5f) 
                : new Color(1, 1, 0, 0.3f);
            
            Gizmos.DrawWireSphere(_detectionSphere.transform.position, sphereRadius);
        }
        #endif
    }
    
    public class CutSphereCollisionDetector : MonoBehaviour
    {
        private CutPatternDetector _cutDetector;
        
        public void Initialize(CutPatternDetector detector) => _cutDetector = detector;
        
        private void OnTriggerEnter(Collider other)
        {
            var enemy = other.GetComponent<Enemy>();
            if (enemy != null) _cutDetector?.OnEnemyEnter(enemy);
        }
    }
}
