using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.InputSystem;

namespace Game
{
    public enum PatternType
    {
        Tap,
        Circle,
        Cut
    }

    public class CursorController : MonoBehaviour
    {
        [Header("Camera setting")]
        [Tooltip("Sélectionnera automatiquement la caméra en fonction de la plateforme")]
        [SerializeField] Camera camMobile;
        [SerializeField] Camera camPC;
        [SerializeField] float planeDistance = 2f;

        private Vector3 _prevPosWs;
        private Vector3 _currPosWs;
        
        private Vector2 _screenPos;

        private Camera _cam;
        private bool _isSlicing;
        private TrailRenderer _trail;

        private CirclePatternDetector _circleDetector;
        private CutPatternDetector _cutDetector;
        private TapPatternDetector _tapDetector;

        private void Awake()
        {
            _trail = GetComponentInChildren<TrailRenderer>();
            if (_trail) _trail.alignment = LineAlignment.View;

            _circleDetector = GetComponent<CirclePatternDetector>();
            _tapDetector = GetComponent<TapPatternDetector>();
            _cutDetector = GetComponent<CutPatternDetector>();

            // Choisir la caméra en fonction de la plateforme
            if (Application.isEditor)
            {
                _cam = GameObject.Find("Cam-Mobile").gameObject.activeInHierarchy ? camMobile : camPC;
            }
            else
            {
                if (Application.isMobilePlatform)
                {
                    camMobile.gameObject.SetActive(true);
                    camPC.gameObject.SetActive(false);
                    _cam = camMobile;
                }
                else
                {
                    camMobile.gameObject.SetActive(false);
                    camPC.gameObject.SetActive(true);
                    _cam = camPC;
                }
            }

            if (_cam == null)
                Debug.LogError("CursorController: aucune caméra assignée !");
            else
                Debug.Log($"CursorController utilise la caméra: {_cam.name} (pixelRect={_cam.pixelRect})");
        }

        public void OnPointerPosition(InputValue value)
        {
            if (_cam == null) return;

            Vector2 screenPos = value.Get<Vector2>();

            // Conversion en espace pixel de la caméra
            Rect pr = _cam.pixelRect;
            Vector2 camPix = screenPos - pr.min;
            _screenPos = camPix;
            Vector3 world = _cam.ScreenToWorldPoint(new Vector3(camPix.x, camPix.y, planeDistance));
            transform.position = world;
        }

        public void OnPointerPress(InputValue value)
        {
            //Debug.Log("OnPointerPress called.");

            // Réinitialiser l'état du cercle AVANT toute nouvelle détection.
            if (_circleDetector == null) Debug.LogWarning("CircleDetector is NULL in CursorController");
            _circleDetector?.OnBeginSlice();
        
            if(_cutDetector == null) Debug.LogWarning("CutDetector is NULL in CursorController");
            _cutDetector?.OnBeginSlice();
        
            // Enregistrer la position de départ pour le tap detector
            if (_tapDetector == null) Debug.LogWarning("TapDetector is NULL in CursorController");
            _tapDetector?.OnBeginPress(transform.position);

            _isSlicing = true;
            BeginSlice();
        }

        public void OnPointerRelease(InputValue value)
        {
            _isSlicing = false;
            Debug.Log("OnPointerRelease called.");
            // Détecter le tap au relâchement de la souris
            _tapDetector?.PerformTap(_screenPos, transform.position);
            
            EndSlice();
        }

        void BeginSlice()
        {
            //Debug.Log("Begin Slice");
            if (_trail == null) return;
            _trail.Clear();
            // On réactive le trail
            _trail.emitting = true;
        }

        void EndSlice()
        {
            //Debug.Log("End Slice");
            if (_trail == null) return;
            _trail.emitting = false;
            _circleDetector?.OnEndSlice();

            if (!CircleFormedThisSlice)
            {
                _cutDetector?.ProcessCut();
            }
        }



        void Update()
        {
            if (_cam == null) return;

            // on mémorise la position actuelle pour la vitesse
            _currPosWs = transform.position;

            if (_isSlicing)
            {
                _circleDetector?.Detect(_currPosWs, _prevPosWs);
                _cutDetector?.AddPoint(_currPosWs);
            }

            _prevPosWs = _currPosWs;
        }

        public Camera GetCamera()
        {
            return _cam;
        }

        public bool IsSlicing()
        {
            return _isSlicing;
        }

        public bool CircleFormedThisSlice
        { get
        {
            return _circleDetector != null && _circleDetector.CircleFormedThisSlice;
        } }
    
        public void HandleEnemiesCaptured(List<Enemy> enemies, PatternType patternType)
        {
            if (enemies == null) return;
            foreach (var e in enemies)
            {
                if (e == null) continue;
                if (e.capturePattern != patternType)
                {
                    StartCoroutine(e.ShowOutline());
                }
                else
                {
                    e.Capture(); 
                }
            }
        }

        public void HandleEnemiesHit(List<Enemy> enemies, PatternType patternType)
        {
            if (enemies == null) return;
            foreach (var e in enemies)
            {
                if (e == null) continue;
                Debug.Log("Ennemy " + e.name + " hit with pattern " + patternType + ", requires " + e.capturePattern);
                if (e.capturePattern != patternType)
                {
                    StartCoroutine(e.ShowOutline());
                }
                else
                {
                    e.TakeDamage();
                }
            }
        }
        
    }
}