using System.Collections;
using System.Collections.Generic;
using UnityEngine;

namespace Game
{
    public class CirclePatternDetector : MonoBehaviour
    {
        private Camera _cam;
        private TrailRenderer _trail;
        private CursorController _cursorController;
        private bool _canDetectCircle = true;
        private bool _circleFormedThisSlice;

        [Header("Circle Detection")]
        [SerializeField] private float circleDetectionThreshold = 0.8f;
        [SerializeField] private float circleAnimationDuration = 0.5f;

        [Header("Shape Validation")]
        [SerializeField] private float closureDistanceTolerance = 1.0f;
        [SerializeField] private float shapeCircularityTolerance = 0.5f;
        [SerializeField] private float minimumRadius = 0.5f;
        [SerializeField] private float minimumAngleCoverage = 270f;

        public bool CircleFormedThisSlice => _circleFormedThisSlice;

        private void Awake()
        {
            _cursorController = GetComponent<CursorController>();
            _trail = GetComponentInChildren<TrailRenderer>();
        }

        private void Start() => _cam = _cursorController.GetCamera();

        public void OnBeginSlice() => _circleFormedThisSlice = false;

        public void OnEndSlice()
        {
            if (_circleFormedThisSlice) return;

            var trailPoints = GetTrailPoints();
            if (trailPoints.Count < 10) return;

            float closureDistance = Vector3.Distance(trailPoints[0], trailPoints[^1]);
            if (closureDistance < closureDistanceTolerance)
            {
                TryDetectUnclosedCircle(trailPoints);
            }
        }

        public void Detect(Vector3 currentPosition, Vector3 previousPosition)
        {
            if (_trail == null || _circleFormedThisSlice || !_canDetectCircle) return;

            var trailPoints = GetTrailPoints();
            if (trailPoints.Count < 4) return;

            if (CheckTrailIntersection(trailPoints))
            {
                _circleFormedThisSlice = true;
                Debug.Log("Forme de cercle fermée détectée !");

                var (center, radius) = CalculateCenterAndRadius(trailPoints);
                var capturedEnemies = GetEnemiesInCircle(center, radius);

                StartCoroutine(AnimateAndShrinkCircle(trailPoints, center, radius, capturedEnemies));
                StartCoroutine(CircleDetectionCooldown());
            }
        }

        private List<Vector3> GetTrailPoints()
        {
            var points = new List<Vector3>();
            int count = _trail.positionCount;
            for (int i = 0; i < count; i++)
            {
                points.Add(_trail.GetPosition(i));
            }
            return points;
        }

        private void TryDetectUnclosedCircle(List<Vector3> points)
        {
            var (center, radius) = CalculateCenterAndRadius(points);
            
            if (radius < minimumRadius)
            {
                _trail.Clear();
                return;
            }

            float variance = CalculateRadiusVariance(points, center, radius);
            if (variance >= shapeCircularityTolerance)
            {
                _trail.Clear();
                return;
            }

            float totalAngle = CalculateTotalAngle(points, center);
            if (totalAngle < minimumAngleCoverage)
            {
                _trail.Clear();
                return;
            }

            _circleFormedThisSlice = true;
            var capturedEnemies = GetEnemiesInCircle(center, radius);
            StartCoroutine(AnimateAndShrinkCircle(points, center, radius, capturedEnemies));
            StartCoroutine(CircleDetectionCooldown());
            _trail.Clear();
        }

        private (Vector3 center, float radius) CalculateCenterAndRadius(List<Vector3> points)
        {
            Vector3 center = Vector3.zero;
            foreach (var point in points)
                center += point;
            center /= points.Count;

            float radius = 0f;
            foreach (var point in points)
                radius += Vector3.Distance(point, center);
            radius /= points.Count;

            return (center, radius);
        }

        private float CalculateRadiusVariance(List<Vector3> points, Vector3 center, float avgRadius)
        {
            float variance = 0f;
            foreach (var point in points)
            {
                float diff = Vector3.Distance(point, center) - avgRadius;
                variance += diff * diff;
            }
            return variance / points.Count;
        }

        private float CalculateTotalAngle(List<Vector3> points, Vector3 center)
        {
            float totalAngle = 0f;
            for (int i = 0; i < points.Count - 1; i++)
            {
                Vector3 dir1 = (points[i] - center).normalized;
                Vector3 dir2 = (points[i + 1] - center).normalized;
                totalAngle += Vector3.Angle(dir1, dir2);
            }
            return totalAngle;
        }

        private bool CheckTrailIntersection(List<Vector3> points)
        {
            for (int i = 0; i < points.Count - 3; i++)
            {
                for (int j = i + 2; j < points.Count - 1; j++)
                {
                    if (DoSegmentsIntersect(points[i], points[i + 1], points[j], points[j + 1]))
                        return true;
                }
            }
            return false;
        }

        private bool DoSegmentsIntersect(Vector3 p1, Vector3 p2, Vector3 p3, Vector3 p4)
        {
            Vector2 a1 = new(p1.x, p1.y);
            Vector2 a2 = new(p2.x, p2.y);
            Vector2 b1 = new(p3.x, p3.y);
            Vector2 b2 = new(p4.x, p4.y);

            Vector2 dirA = a2 - a1;
            Vector2 dirB = b2 - b1;
            float cross = dirA.x * dirB.y - dirA.y * dirB.x;
            
            if (Mathf.Abs(cross) < 0.0001f) return false;

            Vector2 diff = b1 - a1;
            float t = (diff.x * dirB.y - diff.y * dirB.x) / cross;
            float u = (diff.x * dirA.y - diff.y * dirA.x) / cross;

            return t >= 0 && t <= 1 && u >= 0 && u <= 1;
        }

        private List<Enemy> GetEnemiesInCircle(Vector3 center, float radius)
        {
            var enemiesInCircle = new List<Enemy>();
            Vector3 screenCenter = _cam.WorldToScreenPoint(center);
            Vector3 screenEdge = _cam.WorldToScreenPoint(center + Vector3.right * radius);
            float screenRadius = Vector2.Distance(screenCenter, screenEdge);

            foreach (var enemy in Enemy.Enemies)
            {
                if (enemy == null) continue;
                
                Vector3 screenEnemyPos = _cam.WorldToScreenPoint(enemy.transform.position);
                if (Vector2.Distance(screenCenter, screenEnemyPos) <= screenRadius)
                {
                    enemiesInCircle.Add(enemy);
                }
            }
            
            return enemiesInCircle;
        }

        private IEnumerator CircleDetectionCooldown()
        {
            _canDetectCircle = false;
            _trail.emitting = false;
            yield return new WaitForEndOfFrame();
            _trail.Clear();

            yield return new WaitForSeconds(0.5f);

            _canDetectCircle = true;
            if (_cursorController.IsSlicing())
            {
                _trail.emitting = true;
            }
        }

        private IEnumerator AnimateAndShrinkCircle(List<Vector3> initialPoints, Vector3 circleCenter, float circleRadius, List<Enemy> capturedEnemies)
        {
            _cursorController.HandleEnemiesCaptured(capturedEnemies, PatternType.Circle);

            GameObject animCircle = new("AnimatedCircle");
            LineRenderer lineRenderer = SetupLineRenderer(animCircle, initialPoints);

            Vector3 targetCenter = CalculateTargetCenter(capturedEnemies, circleCenter);
            yield return AnimateCircleShrink(lineRenderer, initialPoints, circleCenter, circleRadius, targetCenter);

            if (capturedEnemies.Count > 0)
            {
                _cursorController.HandleEnemiesHit(capturedEnemies, PatternType.Circle);
            }

            Destroy(animCircle);
        }

        private LineRenderer SetupLineRenderer(GameObject obj, List<Vector3> points)
        {
            LineRenderer lr = obj.AddComponent<LineRenderer>();
            
            if (_trail != null)
            {
                lr.material = _trail.material;
                lr.startColor = _trail.startColor;
                lr.endColor = _trail.endColor;
                lr.startWidth = _trail.startWidth;
                lr.endWidth = _trail.endWidth;
            }
            
            lr.useWorldSpace = true;
            lr.positionCount = points.Count;
            lr.SetPositions(points.ToArray());
            lr.loop = true;
            
            return lr;
        }

        private Vector3 CalculateTargetCenter(List<Enemy> enemies, Vector3 fallbackCenter)
        {
            if (enemies.Count == 0) return fallbackCenter;
            
            Vector3 center = Vector3.zero;
            foreach (var enemy in enemies)
                center += enemy.transform.position;
            
            return center / enemies.Count;
        }

        private IEnumerator AnimateCircleShrink(LineRenderer lr, List<Vector3> initialPoints, Vector3 startCenter, float startRadius, Vector3 targetCenter)
        {
            float elapsedTime = 0f;
            var currentPoints = new Vector3[initialPoints.Count];

            while (elapsedTime < circleAnimationDuration)
            {
                elapsedTime += Time.deltaTime;
                float t = Mathf.SmoothStep(0, 1, elapsedTime / circleAnimationDuration);

                float currentRadius = Mathf.Lerp(startRadius, 0f, t);
                Vector3 currentCenter = Vector3.Lerp(startCenter, targetCenter, t);

                for (int i = 0; i < initialPoints.Count; i++)
                {
                    float angle = (float)i / initialPoints.Count * 360f * Mathf.Deg2Rad;
                    Vector3 circlePoint = currentCenter
                        + _cam.transform.right * (Mathf.Cos(angle) * currentRadius)
                        + _cam.transform.up * (Mathf.Sin(angle) * currentRadius);
                    currentPoints[i] = Vector3.Lerp(initialPoints[i], circlePoint, t);
                }
                
                lr.SetPositions(currentPoints);
                yield return null;
            }
        }
    }
}
