﻿Shader "Roystan/Toon/Water"
{
    Properties
    {	
        _DepthGradientShallow("Depth Gradient Shallow", Color) = (0.325, 0.807, 0.971, 0.725)
        _DepthGradientDeep("Depth Gradient Deep", Color) = (0.086, 0.407, 1, 0.749)
        _DepthMaxDistance("Depth Maximum Distance", Float) = 1

        _SurfaceNoise("Surface Noise", 2D) = "white" {}
        _SurfaceNoiseCutoff("Surface Noise Cutoff", Range(0, 1)) = 0.777
        _SurfaceNoiseScroll("Surface Noise Scroll Amount", Vector) = (0.03, 0.03, 0, 0)

        _FoamDistance("Foam Distance", Float) = 0.4

        _SurfaceDistortion("Surface Distortion", 2D) = "white" {}
        _SurfaceDistortionAmount("Surface Distortion Amount", Range(0, 1)) = 0.27

        _Amplitude("Amplitude", Float) = 0.5
        _Speed("Speed", Float) = 1.0
        _Intensity("Intensity", Float) = 1.0
        _Direction("Direction", Vector) = (1, 0, 0, 0)

        // Additive texture properties
        _AdditiveTexture("Additive Texture", 2D) = "black" {}
        _AdditiveStrength("Additive Strength", Range(0, 1)) = 0.5
    }
    SubShader
    {
        Pass
        {
			CGPROGRAM
            #define SMOOTHSTEP_AA 0.01
            #pragma vertex vert
            #pragma fragment frag

            #include "UnityCG.cginc"

            struct appdata
            {
                float4 vertex : POSITION;
                float4 uv : TEXCOORD0;
            };

            struct v2f
            {
                float4 vertex : SV_POSITION;
                float4 screenPosition : TEXCOORD2;
                float2 noiseUV : TEXCOORD0;
                float2 distortUV : TEXCOORD1;
                float2 additiveUV : TEXCOORD3; // UV for additive texture
            };

            float _Amplitude;
            float _Speed;
            float _Intensity;
            float4 _Direction;

            sampler2D _SurfaceNoise;
            float4 _SurfaceNoise_ST;
            sampler2D _SurfaceDistortion;
            float4 _SurfaceDistortion_ST;

            sampler2D _AdditiveTexture;
            float4 _AdditiveTexture_ST;

            float _SurfaceDistortionAmount;
            float _AdditiveStrength;

            v2f vert (appdata v)
            {
                v2f o;

                v.vertex.y += _SinTime.w * _Amplitude * cos(_Time.z * v.uv.y * _Speed);

                o.vertex = UnityObjectToClipPos(v.vertex);
                o.screenPosition = ComputeScreenPos(o.vertex);
                o.noiseUV = TRANSFORM_TEX(v.uv, _SurfaceNoise);
                o.distortUV = TRANSFORM_TEX(v.uv, _SurfaceDistortion);
                o.additiveUV = TRANSFORM_TEX(v.uv, _AdditiveTexture);

                return o;
            }

            float4 _DepthGradientShallow;
            float4 _DepthGradientDeep;

            float _DepthMaxDistance;
            float _SurfaceNoiseCutoff;
            float _FoamDistance;
            float2 _SurfaceNoiseScroll;

            sampler2D _CameraDepthTexture;

            float4 frag (v2f i) : SV_Target
            {
                // Calculate water depth color
                float existingDepth01 = tex2Dproj(_CameraDepthTexture, UNITY_PROJ_COORD(i.screenPosition)).r;
                float existingDepthLinear = LinearEyeDepth(existingDepth01);

                float depthDifference = existingDepthLinear - i.screenPosition.w;

                float waterDepthDifference01 = saturate(depthDifference / _DepthMaxDistance);
                float4 waterColor = lerp(_DepthGradientShallow, _DepthGradientDeep, waterDepthDifference01);

                // Surface noise
                float2 distortSample = (tex2D(_SurfaceDistortion, i.distortUV).xy * 2 - 1) * _SurfaceDistortionAmount;
                float2 noiseUV = float2((i.noiseUV.x + _Time.y * _SurfaceNoiseScroll.x) + distortSample.x, (i.noiseUV.y + _Time.y * _SurfaceNoiseScroll.y) + distortSample.y);
                float surfaceNoiseSample = tex2D(_SurfaceNoise, noiseUV).r;

                // Foam
                float foamDepthDifference01 = saturate(depthDifference / _FoamDistance);
                float surfaceNoiseCutoff = foamDepthDifference01 * _SurfaceNoiseCutoff;

                float surfaceNoise = smoothstep(surfaceNoiseCutoff - SMOOTHSTEP_AA, surfaceNoiseCutoff + SMOOTHSTEP_AA, surfaceNoiseSample);

                // Additive texture
                float4 additiveSample = tex2D(_AdditiveTexture, i.additiveUV) * _AdditiveStrength;

                // Final color
                return waterColor + surfaceNoise + additiveSample;
            }
            ENDCG
        }
    }
}
