using System.Collections;
using Unity.VisualScripting;
using UnityEngine;

public class Fish : MonoBehaviour
{
    private enum FishState
    {
        GROWING,
        SWIMMING,
        JUMPING,
        SHRINKING,
        END
    }

    public bool IsGoodToCatch { get; private set; }
    public int Weight { get; private set; } // In grams

    [Space]
    [SerializeField] private FishSettings fishSettings;
    [SerializeField] private Collider fishCollider;
    [SerializeField] private Animator animator;

    private float jumpDurationWithMultiplier = 0;

    private FishState currentState = FishState.GROWING;
    private bool haveJumped = false;

    private Vector3 targetScale;
    private Vector3 initialScale = Vector3.one;

    private Vector3 targetPosition;
    private Vector3 jumpStartPosition;
    private Vector3 jumpEndPosition;

    private float waterSurfaceHeight;
    private float minX, maxX, minZ, maxZ;
    private Vector3 centerPosition;

    private float swimTime = 0f;
    private float lifeCycleTimer = 0f;
    private float jumpCooldownTimer = 0f;
    private float jumpTimer = 0f;
    private float directionTimer = 0f;

    private static Transform particulesContainer;

    // --------------------------------------------------------------------------------------------
    public void Init(bool isGoodToCatch, float jumpSpeedMultiplier, float minX, float maxX, float minZ, float maxZ, float waterSurfaceHeight)
    {
        this.IsGoodToCatch = isGoodToCatch;

        this.waterSurfaceHeight = waterSurfaceHeight;
        this.minX = minX;
        this.maxX = maxX;
        this.minZ = minZ;
        this.maxZ = maxZ;
        centerPosition = new Vector3((minX + maxX) / 2, transform.position.y, (minZ + maxZ) / 2);

        // Apply the jump speed multiplier
        jumpDurationWithMultiplier = fishSettings.jumpDuration / jumpSpeedMultiplier;
        animator.SetFloat("JumpSpeedMultiplier", jumpSpeedMultiplier);

        // Generate a random scale for the fish
        float randomScaleFactor = Random.Range(fishSettings.minScaleFactor, fishSettings.maxScaleFactor);
        targetScale = new Vector3(randomScaleFactor, randomScaleFactor, randomScaleFactor);

        // Assign a weight to the fish based on his scale
        Weight = Mathf.RoundToInt(Mathf.Lerp(
            fishSettings.minWeight,
            fishSettings.maxWeight,
            (randomScaleFactor - fishSettings.minScaleFactor) / (fishSettings.maxScaleFactor - fishSettings.minScaleFactor)
        ));

        fishCollider.enabled = false;
        initialScale = transform.localScale;
        currentState = FishState.GROWING;

        particulesContainer ??= new GameObject("ParticulesContainer").transform;
    }

    // --------------------------------------------------------------------------------------------
    private void Start()
    {
        ChooseNewTargetPosition();
        transform.localScale = Vector3.zero;
    }

    // --------------------------------------------------------------------------------------------
    private void Update()
    {
        if (currentState != FishState.JUMPING)
        {
            SwimTowardsTarget();

            directionTimer += Time.deltaTime;
            if (directionTimer >= fishSettings.changeDirectionCooldown)
            {
                directionTimer = 0f;
                ChooseNewTargetPosition();
            }

            if (!haveJumped)
            {
                jumpCooldownTimer += Time.deltaTime;
            }
        }

        switch (currentState)
        {
            case FishState.GROWING:
                Grow();
                break;

            case FishState.SWIMMING:
                Swim();
                break;

            case FishState.JUMPING:
                Jump();
                break;

            case FishState.SHRINKING:
                Shrink();
                break;
        }
    }

    // --------------------------------------------------------------------------------------------
    public void Catch()
    {
        if (!IsGoodToCatch)
        {
            Debug.Log("Wrong fish caught, you lost a life!");
        }
        else
        {
            Debug.Log($"Fish of weight {Weight}g caught!");
        }

        Destroy(gameObject);
    }

    // --------------------------------------------------------------------------------------------
    private void Grow()
    {
        lifeCycleTimer += Time.deltaTime;
        float scaleMultiplier = lifeCycleTimer / fishSettings.timeToFullSize;
        transform.localScale = targetScale * scaleMultiplier;

        if (lifeCycleTimer >= fishSettings.timeToFullSize)
        {
            lifeCycleTimer = 0f;
            swimTime = Random.Range(fishSettings.minSwimTime, fishSettings.maxSwimTime);
            currentState = FishState.SWIMMING;
        }
    }

    // --------------------------------------------------------------------------------------------
    private void Shrink()
    {
        lifeCycleTimer += Time.deltaTime;
        float scaleMultiplier = 1 - (lifeCycleTimer / fishSettings.timeToDisappear);
        transform.localScale = initialScale * scaleMultiplier;

        if (lifeCycleTimer >= fishSettings.timeToDisappear)
        {
            lifeCycleTimer = 0f;
            currentState = FishState.END;

            // TODO do smth better than just destroying the fish
            Destroy(gameObject);
        }
    }

    // --------------------------------------------------------------------------------------------
    private void Swim()
    {
        lifeCycleTimer += Time.deltaTime;

        Vector3 distanceToCenter = centerPosition - transform.position;
        if (!haveJumped && jumpCooldownTimer >= fishSettings.jumpCooldown && distanceToCenter.magnitude <= fishSettings.jumpProximity)
        {
            InitJump();
        }

        if (lifeCycleTimer >= swimTime)
        {
            lifeCycleTimer = 0f;
            currentState = FishState.SHRINKING;
        }
    }

    // --------------------------------------------------------------------------------------------
    private void InitJump()
    {
        currentState = FishState.JUMPING;

        haveJumped = true;
        fishCollider.enabled = true;
        animator.Play("Jump");

        Vector3 fxPosition = new Vector3(transform.position.x, waterSurfaceHeight + fishSettings.fxHeight, transform.position.z);
        ParticleSystem jumpFX = Instantiate(fishSettings.jumpFXPrefab, fxPosition, Quaternion.identity, particulesContainer);
        DestroyFXAfterAnimation(jumpFX);

        jumpStartPosition = transform.position;
        jumpEndPosition = jumpStartPosition + transform.forward * 0.5f;

        jumpTimer = 0f;
    }

    // --------------------------------------------------------------------------------------------
    private void Jump()
    {
        jumpTimer += Time.deltaTime;

        float jumpProgress = jumpTimer / jumpDurationWithMultiplier;
        float height = Mathf.Sin(jumpProgress * Mathf.PI) * fishSettings.jumpHeight;

        transform.position = Vector3.Lerp(jumpStartPosition, jumpEndPosition, jumpProgress) + new Vector3(0, height, 0);

        if (jumpTimer >= jumpDurationWithMultiplier)
        {
            EndJump();
        }
    }

    // --------------------------------------------------------------------------------------------
    private void EndJump()
    {
        jumpTimer = 0f;
        lifeCycleTimer = 0f;
        fishCollider.enabled = false;

        Vector3 fxPosition = new Vector3(transform.position.x, waterSurfaceHeight + fishSettings.fxHeight, transform.position.z);
        ParticleSystem landFX = Instantiate(fishSettings.landFXPrefab, fxPosition, Quaternion.identity, particulesContainer);
        DestroyFXAfterAnimation(landFX);

        currentState = FishState.SHRINKING;
    }

    // --------------------------------------------------------------------------------------------
    private void ChooseNewTargetPosition()
    {
        float rangeX = Mathf.Clamp(Random.Range(minX, maxX), minX, maxX);
        float rangeZ = Mathf.Clamp(Random.Range(minZ, maxZ), minZ, maxZ);

        targetPosition = new Vector3(rangeX, transform.position.y, rangeZ);
    }

    // --------------------------------------------------------------------------------------------
    private void SwimTowardsTarget()
    {
        Vector3 direction = (targetPosition - transform.position).normalized;
        Quaternion targetRotation = Quaternion.LookRotation(direction);
        transform.rotation = Quaternion.Slerp(transform.rotation, targetRotation, Time.deltaTime * fishSettings.turnSpeed);

        transform.position += transform.forward * fishSettings.swimSpeed * Time.deltaTime;

        transform.position = new Vector3(transform.position.x, waterSurfaceHeight, transform.position.z);
    }

    // --------------------------------------------------------------------------------------------
    private void DestroyFXAfterAnimation(ParticleSystem fx)
    {
        Destroy(fx.gameObject, fx.main.duration + fx.main.startLifetime.constantMax);
    }

    // --------------------------------------------------------------------------------------------
}
