using UnityEngine;

public class S_CatMovement : MonoBehaviour
{
    [Header("References")]
    public LineRenderer lr;
    public Transform gunTip, cam, player;
    private Vector3 currentGrapplePosition;

    [Header("Dplacement")]
    public float acceleration = 50f;
    public float maxSpeed = 10f;
    public float dragOnGround = 4f;

    [Header("Saut")]
    public float jumpForce = 5.0f;

    [Header("Rotation")]
    public float torqueStrength = 50f;
    public float rotationDamping = 4f;
    public float driftThreshold = 1.5f;

    private Rigidbody rb;
    private bool isOnGround = true;
    private Vector3 inputDir;

    [Header("Particles")]
    ParticleSystem.EmissionModule emissionModule;
    public GameObject driftParticles;
    public Animator anim;

    [Header("Swinging")]
    [SerializeField] private float maxSwingDistance = 25f;
    Vector3 swingPoint;
    SpringJoint joint;

    Vector3 startPosition;

	private void Awake()
	{
		startPosition = transform.position;
	}

	void Start()
    {
        rb = GetComponent<Rigidbody>();

        rb.linearDamping = dragOnGround;
        rb.angularDamping = 2f;

        emissionModule = driftParticles.GetComponent<ParticleSystem>().emission;
    }

	private void OnEnable()
	{
		transform.position = startPosition;
	}

	void Update()
    {
        float h = Input.GetAxisRaw("Horizontal");
        float v = Input.GetAxisRaw("Vertical");
        inputDir = new Vector3(h, 0, v).normalized;

        if (Input.GetKeyDown(KeyCode.Space) && isOnGround)
        {
            rb.AddForce(Vector3.up * jumpForce, ForceMode.Impulse);
            isOnGround = false;
        }

        if (Input.GetKeyDown(KeyCode.LeftAlt))
        {
            StartSwing();
        }
        if (Input.GetKeyUp(KeyCode.LeftAlt))
        {
            StopSwing();
        }

        UpdateVisuals();
    }

    void FixedUpdate()
    {
        MoveCharacter();

        RotateCharacter();

        DrawRope();
    }

    void StartSwing()
    {
        RaycastHit hit;
        if (Physics.Raycast(transform.position, transform.right + transform.forward, out hit, maxSwingDistance))
        {
            Debug.Log("Swinging");
            swingPoint = hit.point;
            joint = gameObject.AddComponent<SpringJoint>();
            joint.autoConfigureConnectedAnchor = false;
            joint.connectedAnchor = swingPoint;

            float distanceFromPoint = Vector3.Distance(transform.position, swingPoint);

            joint.maxDistance = distanceFromPoint * 0.8f;
            joint.minDistance = distanceFromPoint * 0.25f;

            joint.spring = 4.5f;
            joint.damper = 7f;
            joint.massScale = 4.5f;

            lr.positionCount = 2;
            currentGrapplePosition = transform.position;
        }
    }

    void StopSwing()
    {
        lr.positionCount = 0;
        Destroy(joint);
    }


    void MoveCharacter()
    {
        if (inputDir.magnitude > 0)
        {
            rb.AddForce(inputDir * acceleration, ForceMode.Acceleration);
        }

        // Limitation de vitesse (Clamp)
        Vector3 flatVel = new Vector3(rb.linearVelocity.x, 0, rb.linearVelocity.z);
        if (flatVel.magnitude > maxSpeed)
        {
            Vector3 cappedVel = flatVel.normalized * maxSpeed;
            rb.linearVelocity = new Vector3(cappedVel.x, rb.linearVelocity.y, cappedVel.z);
        }
    }

    void RotateCharacter()
    {
        if (inputDir.magnitude == 0) return;

        // Calcule la rotation cible
        Quaternion targetRotation = Quaternion.LookRotation(inputDir);

        // Calcule la diffrence d'angle entre la rotation actuelle et la cible
        // On veut tourner autour de l'axe Y (Vector3.up)
        float angleDiff = Vector3.SignedAngle(transform.forward, inputDir, Vector3.up);

        // FORMULE PID SIMPLIFIE POUR LA ROTATION :
        // Force = (Erreur * Force) - (VitesseActuelle * Amortissement)
        // Cela permet de tourner vite vers la cible, mais de freiner la rotation en arrivant pour ne pas dpasser.

        float torqueToApply = (angleDiff * torqueStrength) - (rb.angularVelocity.y * rotationDamping);

        rb.AddTorque(Vector3.up * torqueToApply, ForceMode.Acceleration);
    }

    void UpdateVisuals()
    {
        if (inputDir.magnitude > 0 && isOnGround)
        {
            anim.Play("Walk2");
        }
        else
        {
            anim.Play("Idle");
        }

        if (Mathf.Abs(rb.angularVelocity.y) > driftThreshold && isOnGround)
        {
            emissionModule.enabled = true;
        }
        else
        {
            emissionModule.enabled = false;
        }

        if (inputDir.magnitude < 0.1f && isOnGround && rb.linearVelocity.magnitude > 0.5f)
        {
            emissionModule.enabled = true;
        }
        else if (inputDir.magnitude < 0.1f && isOnGround && rb.linearVelocity.magnitude < 0.5f)
        {
            emissionModule.enabled = false;
        }
    }

    private void OnCollisionEnter(Collision collision)
    {
        if (collision.gameObject.CompareTag("Ground"))
        {
            isOnGround = true;
        }
    }

    void DrawRope()
    {
        if (!joint) return;

        currentGrapplePosition = Vector3.Lerp(currentGrapplePosition, swingPoint, Time.deltaTime * 8f);

        lr.SetPosition(0, gunTip.position);
        lr.SetPosition(1, swingPoint);
    }
}