﻿using UnityEngine;
using System.Collections;
using System.Collections.Generic;
using System;

public class ShadowMapSpawner : MonoBehaviour
{
    public enum BlockType
    {
        Start,      //0
        End,        //1
        Plateforme, //2
        Collider,   //3
        Voiture,    //4
        Vide,       //5
        Count       //6
    }

    [Serializable]
    public struct TypedPrefab
    {
        public BlockType type;
        public GameObject prefab;
    }
    [SerializeField]
    public TypedPrefab[] m_TypedPrefabs;
    public Dictionary<BlockType, List<GameObject>> m_PrefabsDictionary;


    [Serializable]
    public struct TypedColor
    {
        public BlockType type;
        public Color color;
    }
    [SerializeField]
    public TypedColor[] m_TypedColors;
    public Dictionary<Color, BlockType> m_ColorDictionary;

    public enum GameType
    {
        XZPlane,
        XYPlane,
        TPS
    }

    public GameType m_CurrentSceneType;

    public Texture2D m_ShadowMap;
    
    public float m_ScaleFactor;

    // Use this for initialization
    void Start ()
    {
        //On crée le dictionaire pour pouvoir le remplir
        m_PrefabsDictionary = new Dictionary<BlockType, List<GameObject>>();
        //Remplit le dictionaire avec chaque élément du tableau renseignés dans l'Inspector
        foreach (TypedPrefab typedPrefab in m_TypedPrefabs)
        {
            //Si le dictionaire  contient déja des prefabs de ce type
            if(m_PrefabsDictionary.ContainsKey(typedPrefab.type))
            {
                //On ajoute le prefab à la liste de prefab existante de ce type
                m_PrefabsDictionary[typedPrefab.type].Add(typedPrefab.prefab);
            }
            else
            {
                //On Ajoute une nouvelle liste de prefab de ce type au dictionaire
                m_PrefabsDictionary.Add(typedPrefab.type, new List<GameObject>());

                //On Ajoute le prefab à la nouvelle liste
                m_PrefabsDictionary[typedPrefab.type].Add(typedPrefab.prefab);
            }
        }


        m_ColorDictionary = new Dictionary<Color, BlockType>();
        foreach (TypedColor typedColor in m_TypedColors)
        {
            m_ColorDictionary.Add(typedColor.color, typedColor.type);
        }


        for (int pixelXID = 0; pixelXID < m_ShadowMap.width; pixelXID++)
        {
            for (int pixelYID = 0; pixelYID < m_ShadowMap.height; pixelYID++)
            {
                InstantiateBlockByColor(m_ShadowMap.GetPixel(pixelXID, pixelYID), pixelXID, pixelYID);
            }

        }
	}

    void InstantiateBlockByColor(Color blockColor, int xPosition, int yPosition)
    {
        int blockID = GetBlockTypeByColor(blockColor);
        if(blockID < 0)
        {
            Debug.LogError("Wrong Color in shadowMap detected on pixel : " + xPosition + "/" + yPosition);
            return;
        }
        if(blockID == 5)
            return;

        Vector3 blockWorldPosition = GetBlockPositionByPixelPosition(xPosition, yPosition, m_CurrentSceneType);
        if(blockID == (int)BlockType.Collider)
            blockWorldPosition += Vector3.up;

        //On récupère la liste de prefab associé au type de block correspondant à la couleur du pixel
        List<GameObject> prefabsType = m_PrefabsDictionary[(BlockType)blockID];
        int randomID = 0;
        if(prefabsType.Count > 1)
            randomID = UnityEngine.Random.Range(0, prefabsType.Count);


        GameObject newBlock = GameObject.Instantiate(prefabsType[randomID], blockWorldPosition, Quaternion.identity) as GameObject;
        newBlock.transform.SetParent(transform, true);
    }

    int GetBlockTypeByColor(Color blockColor)
    {
        BlockType blockID;
        //Recupère le type de bloc associé à la couleur "blockColor" s'il en existe une
        if(m_ColorDictionary.TryGetValue(blockColor, out blockID))
        {
            return (int)blockID;
        }
        //sinon renvoi -1
        return -1;
    }
    Vector3 GetBlockPositionByPixelPosition(int xPosition, int yPosition, GameType gameType)
    {
        Vector3 worldPosition = Vector3.zero;
        switch (gameType)
        {
            case GameType.XZPlane:
            case GameType.TPS:
                {
                    worldPosition = new Vector3(xPosition, 0, yPosition);
                }
                break;
            case GameType.XYPlane:
                {
                    worldPosition = new Vector3(xPosition, yPosition, 0);
                }
                break;
        }
        return worldPosition * m_ScaleFactor;
    }

}
