﻿using UnityEngine;
using System.Collections;
using System.Collections.Generic;

public class LevelManager : MonoBehaviour
{
    private static LevelManager s_Instance;
    public static LevelManager Instance
    {
        set { s_Instance = value; }
        get {
                return s_Instance;
            }
    }

    public enum GameType
    {
        XZPlane,
        XYPlane,
        TPS
    }
    public enum BlockType
    {
        Start,      //0
        End,        //1
        Plateforme, //2
        Collider,   //3
        Voiture,    //4
        Vide,       //5
        Arbre,      //6
        Fruit,      //7
        Verrou,     //8
        Porte,      //9
        Main,       //10
        Crane,      //11
        Portail1,   //12
        Portail2,   //13
        Mine1,      //14
        Mine2,      //15

        Count
    }

    public Texture2D[] phase_Levels;
    //public Texture2D[] phase2_Levels;
    //public Texture2D[] phase3_Levels;
    
    public int m_CurrentLevelID = 0;
    public static int m_MaxLevel = 3;

    public int m_PhaseID;

    public Transform levelData;
    public Collider m_MouseClickCollider;
    public float m_ScaleFactor;

    [System.Serializable]
    public struct TypedPrefab
    {
        public BlockType type;
        public Color color;
        public GameObject prefab;
    }
    [SerializeField]
    public TypedPrefab[] m_TypedPrefabs;
    public Dictionary<BlockType, List<GameObject>> m_PrefabsDictionary;
    public Dictionary<Color, List<BlockType>> m_ColorDictionary;

    public GameType m_CurrentSceneType;
    public Texture2D m_ShadowMap;
    public void Awake()
    {
        s_Instance = this;
        Init();
    }

    public void Init()
    {
        //On crée le dictionaire pour pouvoir le remplir
        m_PrefabsDictionary = new Dictionary<BlockType, List<GameObject>>();
        m_ColorDictionary = new Dictionary<Color, List<BlockType>>();


        //Remplit le dictionaire avec chaque élément du tableau renseignés dans l'Inspector
        foreach (TypedPrefab typedPrefab in m_TypedPrefabs)
        {
            //Si le dictionaire  contient déja des prefabs de ce type
            if (m_PrefabsDictionary.ContainsKey(typedPrefab.type))
            {
                //On ajoute le prefab à la liste de prefab existante de ce type
                m_PrefabsDictionary[typedPrefab.type].Add(typedPrefab.prefab);
            }
            else
            {
                //On Ajoute une nouvelle liste de prefab de ce type au dictionaire
                m_PrefabsDictionary.Add(typedPrefab.type, new List<GameObject>());

                //On Ajoute le prefab à la nouvelle liste
                m_PrefabsDictionary[typedPrefab.type].Add(typedPrefab.prefab);
            }

            // Set dictionnary Colors

            if (m_ColorDictionary.ContainsKey(typedPrefab.color))
            {
                m_ColorDictionary[typedPrefab.color].Add(typedPrefab.type);
            }
            else
            {
                Color colorToCompare = typedPrefab.color;

                colorToCompare.r = Mathf.Round(colorToCompare.r * 100.0f) / 100.0f;
                colorToCompare.g = Mathf.Round(colorToCompare.g * 100.0f) / 100.0f;
                colorToCompare.b = Mathf.Round(colorToCompare.b * 100.0f) / 100.0f;

                //On Ajoute une nouvelle liste de prefab de ce type au dictionaire
                m_ColorDictionary.Add(colorToCompare, new List<BlockType>());

                //On Ajoute le prefab à la nouvelle liste
                m_ColorDictionary[colorToCompare].Add(typedPrefab.type);

            }

        }
    }
    //Par default, la création d'un niveau n'est pas un tutorial,
    //le second paramètre permet de manuellement forcer la création d'une map de tuto
    public void CreateLevel(int levelID, bool isNewStart = false)
    {
        m_CurrentLevelID = levelID;

        if (levelID == 0)
            SoundManager.PlayMusic("Onze");

        if (levelID > 0)
            SoundManager.PlayMusic("Sur-mer");

        Debug.Log(phase_Levels[m_CurrentLevelID]);
        m_ShadowMap = phase_Levels[m_CurrentLevelID];

        levelData.gameObject.SetActive(true);
        DestroyLevel();

        m_MouseClickCollider.transform.position = new Vector3(m_ShadowMap.width, -1, m_ShadowMap.height);
        m_MouseClickCollider.transform.localScale = new Vector3(m_ShadowMap.width * 2f, 1, m_ShadowMap.height * 2f);


        Color[] pixelsList = m_ShadowMap.GetPixels();

        for (int pixelXID = 0; pixelXID < m_ShadowMap.width; pixelXID++)
        {
            for (int pixelYID = 0; pixelYID < m_ShadowMap.height; pixelYID++)
            {
                Color colorToCompare = pixelsList[pixelXID + pixelYID * m_ShadowMap.width];

                colorToCompare.r = Mathf.Round(colorToCompare.r * 100.0f) / 100.0f;
                colorToCompare.g = Mathf.Round(colorToCompare.g * 100.0f) / 100.0f;
                colorToCompare.b = Mathf.Round(colorToCompare.b * 100.0f) / 100.0f;

                InstantiateBlockByColor(colorToCompare, pixelXID, pixelYID);
            }

        }
        //Si on est sur un tuto
        if (levelID != 0)
            FindObjectOfType<Robot_Lvl1>().ActiveAllFunctionnalities();
        if(isNewStart)
            FindObjectOfType<Robot_Lvl1>().Light_robot.color = Color.white;
    }

    public void DestroyLevel()
    {
        for (int i = 0; i < levelData.childCount; i++)
        {
            Transform objet = levelData.GetChild(i);
            Destroy(objet.gameObject);
        }
    }
    void InstantiateBlockByColor(Color blockColor, int xPosition, int yPosition)
    {
        int blockID = GetBlockTypeByColor(blockColor);
        if (blockID < 0)
        {
            Debug.LogError("Wrong Color (" + blockColor + ") in shadowMap detected on pixel : " + xPosition + "/" + yPosition);
            return;
        }
        if (blockID == 5)
            return;

        Vector3 blockWorldPosition = GetBlockPositionByPixelPosition(xPosition, yPosition, m_CurrentSceneType);
        if (blockID == (int)BlockType.Collider)
            blockWorldPosition += Vector3.up;

        //On récupère la liste de prefab associé au type de block correspondant à la couleur du pixel
        List<GameObject> prefabsType = m_PrefabsDictionary[(BlockType)blockID];
        int randomID = 0;
        if (prefabsType.Count > 1)
            randomID = UnityEngine.Random.Range(0, prefabsType.Count);

        Quaternion axe;

        if (blockID == 15)
        {
            axe = Quaternion.Euler(0, 90, 0);
        }
        else
        {
            axe = Quaternion.identity;
        }


        GameObject newBlock = GameObject.Instantiate(prefabsType[randomID], blockWorldPosition, axe) as GameObject;
        newBlock.transform.SetParent(levelData, true);
    }

    int GetBlockTypeByColor(Color blockColor)
    {
        List<BlockType> blockIDList;
        //Recupère le type de bloc associé à la couleur "blockColor" s'il en existe une
        if (m_ColorDictionary.TryGetValue(blockColor, out blockIDList))
        {
            return (int)blockIDList[Random.Range(0, blockIDList.Count)];
        }
        //sinon renvoi -1
        return -1;
    }
    Vector3 GetBlockPositionByPixelPosition(int xPosition, int yPosition, GameType gameType)
    {
        Vector3 worldPosition = Vector3.zero;
        switch (gameType)
        {
            case GameType.XZPlane:
            case GameType.TPS:
                {
                    worldPosition = new Vector3(xPosition, 0, yPosition);
                }
                break;
            case GameType.XYPlane:
                {
                    worldPosition = new Vector3(xPosition, yPosition, 0);
                }
                break;
        }
        return worldPosition * m_ScaleFactor;
    }


}
